// Simple CloudWatch metric math republishing function
// Takes as input a GetMetricData call and a template, queries CloudWatch and publishes the queried metric into a new metric
const aws = require('aws-sdk');
const cloudwatch = new aws.CloudWatch();

const MS_IN_SEC = 1000;
const SECS_IN_MIN = 60;

const EXAMPLE_QUERIES = {
    delayMinutes: 4,         // Give this enough so that data is complete
    period: 60,              // in seconds
    queries: [
        {
            gmdParams: {        // Parameters for GetMetricData
                MetricDataQueries: [
                    {
                        Id: 'inv',
                        Expression: 'SUM(SEARCH(\'{AWS/Lambda,FunctionName} MetricName="Invocations"\', \'Sum\', 60))',
                        Label: 'Invocations'
                    },
                    {
                        Id: 'errs',
                        Expression: 'SUM(SEARCH(\'{AWS/Lambda,FunctionName} MetricName="Errors"\', \'Sum\', 60))',
                        Label: 'Errors'
                    }
                ]
            },
            metricTemplate: {   // How the metrics should be published, a Dimension value or MetricName should be set to $label, to pick up response from GMD
                Namespace: 'LambdaAggregates',
                Dimensions: [
                    { Name: 'Function', Value: 'ALL' }
                ],
                MetricName: '$label'
            }
        }
    ]
};

const getAndRepublish = async (query, startInSecs, endInSecs) => {
    const gmdParams = Object.assign({}, query.gmdParams, { StartTime: startInSecs, EndTime: endInSecs});
    const metricTemplate = query.metricTemplate;

    const response = await cloudwatch.getMetricData(gmdParams).promise();
    const results = response.MetricDataResults || [];
    let resultsFound = false;
    
    results.forEach(result => {
        if (result.Values && result.Values.length && metricTemplate) {
            const label = result.Label;
            const value = result.Values[0];
            const metric = {};
            let metricTemplateStr = JSON.stringify(metricTemplate);

            metricTemplateStr = metricTemplateStr.replace(/[$]label/g, label);
            const replacedMetricTemplate = JSON.parse(metricTemplateStr);
            const dimensions = [];

            resultsFound = true;
            replacedMetricTemplate.Dimensions.forEach(dim => {
                dimensions.push(dim.Name);
                metric[dim.Name] = dim.Value;
            });

            metric[replacedMetricTemplate.MetricName] = value;
            metric._aws = {
                Timestamp:  startInSecs * MS_IN_SEC,
                CloudWatchMetrics: [
                    {
                        Namespace: replacedMetricTemplate.Namespace,
                        Dimensions: [ dimensions ],
                        Metrics: [ { Name: replacedMetricTemplate.MetricName }]
                    }
                ]
            };
            
            console.log(JSON.stringify(metric));
        }
    });
    
    if (!resultsFound) {
        console.log(`No results from GetMetricData for query ${JSON.stringify(gmdParams, null, 4)}. Response was ${JSON.stringify(response, null, 4)}`);
    } 
};

exports.handler = async (event) => {
    const nowInMs = new Date().getTime();
    const nowInSecsFlooredToMin = Math.floor(nowInMs / MS_IN_SEC / SECS_IN_MIN) * SECS_IN_MIN;
    const delayMinutes = event.delayMinutes || 4;
    const period = event.period || 60;
    const startAtSecs = nowInSecsFlooredToMin - delayMinutes * SECS_IN_MIN;
    const endAtSecs = startAtSecs + period;
    const queries = event.queries;

    if (queries) {
        for (let i = 0; i < queries.length; i++) {
            await getAndRepublish(queries[i], endAtSecs - SECS_IN_MIN, endAtSecs);
        }
    } else {
        console.error(`No queries configured, was expecting parameters like so\n${JSON.stringify(EXAMPLE_QUERIES, null, 4)}`);
    }

    return 'done';
};
