from aws_cdk import core, aws_cloudwatch
import boto3
import requests
import yaml
import re
import sys
import logging
import json
import copy

REGION_MAPPING = {
  'ap-southeast-1': [
    'bkk01',
    'hkg01',
    'sel01',
    'sel03',
    'sin01',
    'tpe01',
    'tpe03',
    'tyo01',
  ],

  'eu-west-1': [
    'ams02',
    'ams03',
    'arn03',
    'ber01',
    'cdg02',
    'cph01',
    'fra02',
    'fra05',
    'fra06',
    'hel01',
    'lhr03',
    'lhr04',
    'lhr05',
    'mad01',
    'mil01',
    'mrs01',
    'osl01',
    'prg02',
    'vie01',
    'waw01',
  ],

  'us-east-1': [
    'atl01',
    'dfw02',
    'iad03',
    'iad05',
    'jfk04',
    'jfk06',
    'mia02',
    'qro01',
    'rio01',
    'sao01',
  ],

  'us-east-2': [
    'cmh01',
    'den01',
    'ord02',
    'ord03',
    'ymq01',
    'yto01',
  ],

  'us-west-2': [
    'hou01',
    'lax03',
    'pdx01',
    'phx01',
    'sea01',
    'sjc02',
    'sjc05',
    'slc01',
    'syd01',
  ]

}


logger = logging.getLogger()
logging.basicConfig(stream=sys.stdout, level=logging.INFO)
# instantiate cloudwatch client
cw = boto3.client('cloudwatch')

CW_REGIONS = ['us-west-2', 'us-east-1', 'us-east-2', 'ap-southeast-1', 'eu-west-1']
CONSUL_API = "https://consul.internal.justin.tv"

class CloudwatchDashboardMakerStack(core.Stack):

    def __init__(self, scope: core.Construct, id: str, **kwargs) -> None:
        super().__init__(scope, id, **kwargs)
        consul_dcs = []
        # get list of DC's from consul
        try:
            resp = requests.get("{}/v1/catalog/datacenters".format(CONSUL_API))
            consul_dcs = consul_filter(resp.json())
        except:
            logger.warning("Failed to query consul, automatic pop discovery will not work.")

        dashboard_metadata = []
        # get dashboards that want to be templated
        with open('dashboard-layout.yaml') as f:
            dashboard_metadata = yaml.load(f, Loader=yaml.Loader)

        for db_name, db_config in dashboard_metadata.items():
            try:
                db_data = cw.get_dashboard(DashboardName=db_name)
                self.process_dashboard(scope, id, json.loads(db_data['DashboardBody']), db_name, db_config, consul_dcs)
            except Exception as e:
                logger.error("Failed to get dashboard contents for %s, %s", db_name, e)

    def create_entrypoint_dashboard(self, scope, id, db_name, db_config, generated_db_names):
        entrypoint_json = {"widgets":[{"type":"text","x":0,"y":0,"width":24,"height":12,"properties":{"markdown":""}}]}
        button_template = " [button:primary:{}](#dashboards:name={}) "
        base_md_template = "\n# Entrypoint Dashboard for {0}\n\nThis dashboard encapsulates all generated dashboards from the base dashboard {0}\n\n".format(db_name)
        for gdb in generated_db_names:
            if db_config['TemplatePop'] in gdb:
                base_md_template += button_template.format(gdb, db_name)
            else:
                base_md_template += button_template.format(gdb, gdb)
        entrypoint_json['widgets'][0]['properties']['markdown'] = base_md_template
        # create entrypoint dashboard
        aws_cloudwatch.CfnDashboard(self, "entrypoint_{}".format(db_name), dashboard_name="entrypoint-{}".format(db_name), dashboard_body=json.dumps(entrypoint_json))

    # control logic for taking in a template json and yaml config and making lots of replica
    # dashboards or augmenting an existing one.
    def process_dashboard(self, scope, id, db_json, db_name, db_config, consul_dcs):
        db_final_json = {'widgets': []}
        # Go through each widget and if replication is by TemplateRole, then make the widget multi-region
        for widget in db_json['widgets']:
            widget_to_add = widget
            # If the filtering is by role, make all metrics multi-region.
            if 'TemplateRole' in db_config.keys():
                widget_to_add = multi_region_widget(widget, db_config)
            db_final_json['widgets'].append(widget_to_add)

        # Replace pop list from entire consul list to overriden, manual list if applicable
        if 'Pops' in db_config.keys():
            consul_dcs = db_config['Pops']

        # get list of serialized dashboard json strings to build
        dashboards_to_build = generate_db_configs(db_final_json, db_config, consul_dcs)
        generated_db_names = []
        for db_dimension, db_body in dashboards_to_build.items():
            gdb_name = "{}_cdk_{}".format(db_name, db_dimension)
            generated_db_names.append(gdb_name)
            aws_cloudwatch.CfnDashboard(self, "auto_cdk_{}_{}".format(db_name,db_dimension), dashboard_name=gdb_name, dashboard_body=db_body)


        if 'Entrypoint' in db_config.keys():
            self.create_entrypoint_dashboard(scope, id, db_name, db_config, generated_db_names)


# Get details about the metrics within a widget and return two mappings.
# metrics_dict maps metric ID to entire metric list
# expr_mapping maps metric ID to list of other metrics in the widget it may reference
def get_widget_details(metrics_list):
    # reference dict for ease. Maps metric ID to None or expression string
    metric_expr_dict_ref = {}
    metrics_dict = {}
    expr_mapping = {}
    for metric in metrics_list:
        # The dictionary that contains the metric's ID is always the last item in the list
        metric_id = metric[-1]['id']
        metric_expr_dict_ref[metric_id] = None if 'expression' not in metric[-1].keys() else metric[-1]['expression']
        expr_mapping[metric_id] = []
        metrics_dict[metric_id] = metric

    # go through all expression strings and find reference
    for mId, expression in metric_expr_dict_ref.items():
        if expression is not None:
            # now go through each metric key to see if we can find any metric key (meId) in the expression string
            for meId in metrics_dict.keys():
                if meId in expression:
                    expr_mapping[mId].append(meId)

    return metrics_dict, expr_mapping

# For each metric in a widget, duplicate the same metric in the list of regions. Supports math expressions
def multi_region_widget(widget, db_config):
    # clean list of metrics to remove duplicate expressions
    # (i.e: same metric expressions that appear more than once but may have different metadata like region)
    metrics_dict, expr_mapping = get_widget_details(widget['properties']['metrics'])

    # use either defaults or provided list of regions
    regions = db_config['CWRegions'] if ('CWRegions' in db_config.keys() and len(db_config['CWRegions']) > 0) else CW_REGIONS
    # the list that will replace the metrics in the input widget. Start with the metrics from the default region
    metrics_list = [metric for k, metric in metrics_dict.items()]
    new_metric_map = {}
    new_metric_to_old_metric_map = {}
    old_metric_to_new_metrics_map = {}

    for region in regions:
        # make region metric ID friendly
        rid = region.replace("-","_")
        mIndex = 1
        # actually make the metrics and add them to the final list
        for mid, metric in metrics_dict.items():
            new_metric = copy.deepcopy(metric)
            nmid = "{}_{}".format(rid, mIndex)
            new_metric[-1]['region'] = region
            new_metric[-1]['id'] = nmid
            mIndex += 1
            # populate the maps
            new_metric_map[nmid] = new_metric
            new_metric_to_old_metric_map[nmid] = mid
            if mid in old_metric_to_new_metrics_map.keys():
                old_metric_to_new_metrics_map[mid].append(nmid)
            else:
                old_metric_to_new_metrics_map[mid] = [nmid]

    # go through each region's metrics, replace needed strings with the correct mid for the given region, and add the metric to the final list
    for nmid, new_metric in new_metric_map.items():
        final_new_metric = new_metric
        region_prepend = nmid.rsplit('_', 1)[0]
        # get the list of RAW values that need to be string replaced
        repl_list = expr_mapping[new_metric_to_old_metric_map[nmid]]
        # go through each value that the raw metric includes in its expression
        for raw_mid_to_replace in repl_list:
            # attempt to find what new metric id the raw metric maps to in this given region
            new_mid_as_replacement = [mid for mid in old_metric_to_new_metrics_map[raw_mid_to_replace] if region_prepend in mid]
            if len(new_mid_as_replacement) > 0:
                final_new_metric[-1]['expression'] = final_new_metric[-1]['expression'].replace(raw_mid_to_replace, new_mid_as_replacement[0])

        # always append the new metric in case the list is empty
        metrics_list.append(final_new_metric)

    # replace old widget metrics content with new widgets
    widget['properties']['metrics'] = metrics_list

    return widget

# takes a json object, db_json, serializes and str replaces instances of template_str
# and return the serialized json str
def dashboard_replicate(db_json, template_str, repl):
    return json.dumps(db_json).replace(template_str, repl)

# Find the aws region this pop sends metrics to
def find_region_for_pop(pop):
    default = "us-west-2"
    for region, list_of_pops in REGION_MAPPING.items():
        if pop in list_of_pops:
            return region
    return default

# Given the deserialized dashboard json and metdata, replicate the json according to the replication dimension specified
# Return a dict of the dimension specified to cw dashboard strings
def replicate_dashboard(db_json, db_config, consul_dcs):
    db_json_dict = {}
    # Handle replicating pops if specified
    if 'TemplatePop' in db_config.keys():
        if 'TemplatePopRegion' not in db_config.keys():
            logger.error("You must specify what region your Template dashboard ")
        else:
            for pop in consul_dcs:
                region_for_this_pop = find_region_for_pop(pop)
                # replace the regions in the json with the region this POP uploads metrics to
                db_json_modified = json.loads(dashboard_replicate(db_json, db_config['TemplatePopRegion'], region_for_this_pop))
                # replace the pop in the json with the POP we're going with
                db_json_dict[pop] = dashboard_replicate(db_json_modified, db_config['TemplatePop'], pop)
    # Otherwise, handle replicating roles if specified
    elif 'TemplateRole' in db_config.keys():
        list_of_roles = []
        if 'Roles' in db_config.keys() and len(db_config['Roles']) > 0:
            list_of_roles = db_config['Roles']
        for role in list_of_roles:
            db_json_dict[role] = dashboard_replicate(db_json, db_config['TemplateRole'], role)
    return db_json_dict

# Take a template dashboard json object and generate a list of
# dashboard json strings for the cdk to build
def generate_db_configs(db_json, db_config, consul_dcs):
    dict_of_dashboards = {}
    if 'TemplatePop' in db_config.keys():
        dict_of_dashboards = {db_config['TemplatePop']: json.dumps(db_json)}
    elif 'TemplateRole' in db_config.keys():
        dict_of_dashboards = {db_config['TemplateRole']: json.dumps(db_json)}
    else:
        return dict_of_dashboards
    # cannot have both Role AND Pop replication at the same time. Choose one or the other
    # for implementation simplicity for now and not seeing it as a common usecase.
    if 'TemplateRole' in db_config.keys() and 'TemplatePop' in db_config.keys():
        logger.error("Cannot specify 'TemplateRole' and 'TemplatePop' together in a dashboard configuration!")
        return list_of_dashboards

    # Dashboard per logic is to just gather a list of json objects.
    repl_dbs = replicate_dashboard(db_json, db_config, consul_dcs)
    final_dict_of_dashboards = {**dict_of_dashboards, **repl_dbs}

    return final_dict_of_dashboards



# filters bare metal pops out of list of consul POPs (dumb regex matching)
def consul_filter(dc_list):
    pattern = re.compile("^[a-zA-Z]{3}[0-9]{2}$")
    return [dc for dc in dc_list if pattern.search(dc) is not None]
