#!/usr/bin/env bash

set -eo pipefail

REGIONS=("us-west-2" "us-east-1" "us-east-2" "eu-west-1" "ap-southeast-1")
RETENTION=90
LOGGROUP=""
CONFIGFILE=""
DRYRUN=false

while getopts ":c:d" opt; do
  case ${opt} in
    c )
      CONFIGFILE=$OPTARG
      ;;
    d )
      DRYRUN=true
      ;;
    \? )
      echo "Usage: $0 [ -c configfile ][ -d ]" 1>&2
      exit 1
      ;;
  esac
done


requirements() {
    command -v jq &> /dev/null || ( echo "jq required but missing" && exit 1 )
    command -v aws &> /dev/null || ( echo "aws required but missing" && exit 1 )
}


requirements

CALLER_IDENTITY=$(aws sts get-caller-identity | jq -r .Account)

# Validation to make sure you're using twitch-video-ops-aws
((CALLER_IDENTITY != 277437172290)) && {
  echo "Incorrect caller identity $CALLER_IDENTITY. Please use credentials for the twitch-video-ops-aws account (277437172290)"
  exit 1
}

if [ -z "$CONFIGFILE" ]; then
  echo "No config file specified. Please specify a config file using -c <service> (no need to append '.cfg')"
  exit 1
fi

source "groups/${CONFIGFILE}.cfg" || (echo "Config for ${CONFIGFILE} not found!" && exit 1)

for region in "${REGIONS[@]}"; do
    echo "Adding log group $LOGGROUP to $region"
    $DRYRUN || { aws --region "$region" logs create-log-group --log-group-name $LOGGROUP \
    --tags Source=CloudwatchRetentionPolicy || (true && echo "Continuing..") }
    
    echo "Configuring retention policy of $RETENTION days to $LOGGROUP ($region)"
    $DRYRUN || { aws --region "$region" logs put-retention-policy --log-group-name $LOGGROUP \
    --retention-in-days $RETENTION || (true && echo "Continuing..") }
done


