#!/usr/bin/env python

import sys
import os
import errno
import json
import requests
import argparse


CONSUL_HOST = 'api.us-west-2.prod.consul.live-video.a2z.com'
CONSUL_PORT = 80
INVENTORY_BACKUP_DIR = "inventories"
API_PREFIX = '/v1'
ENDPOINTS = {
        'raft_peers': '{}/operator/raft/configuration'.format(API_PREFIX),
        'datacenters': '{}/catalog/datacenters'.format(API_PREFIX),
        'autopilot_health': '{}/operator/autopilot/health'.format(API_PREFIX),
        }


# Taken from https://stackoverflow.com/a/600612/119527
def mkdir_p(path):
    try:
        os.makedirs(path)
    except OSError as exc:
        if exc.errno == errno.EEXIST and os.path.isdir(path):
            pass
        else:
            raise


def safe_open(path, op):
    ''' Open "path" for writing, creating any parent directories as needed.
    '''
    mkdir_p(os.path.dirname(path))
    return open(path, op)


def is_reachable(params={}):
    r = requests.get('http://{}:{}{}'.format(CONSUL_HOST, CONSUL_PORT, ENDPOINTS['datacenters']), params=params)
    return r.status_code == 200


def get_datacenters(params={}):
    r = requests.get('http://{}:{}{}'.format(CONSUL_HOST, CONSUL_PORT, ENDPOINTS['datacenters']), params=params)
    r.raise_for_status()
    return r.json()


def get_servers(dc=None, params={}):
    if dc is not None:
        params.update({'dc': dc})
    r = requests.get('http://{}:{}{}'.format(CONSUL_HOST, CONSUL_PORT, ENDPOINTS['raft_peers']), params=params)
    r.raise_for_status()
    return r.json()


def get_failure_tolerance(dc=None, params={}):
    if dc is not None:
        params.update({'dc': dc})
    r = requests.get('http://{}:{}{}'.format(CONSUL_HOST, CONSUL_PORT, ENDPOINTS['autopilot_health']), params=params)
    r.raise_for_status()
    return {'FailureTolerance': r.json().get('FailureTolerance', {})}


def insert_hostvars(inventory, vars={}):
    # create needed _meta key
    inventory['_meta'] = {}
    inventory['_meta']['hostvars'] = {}
    if vars:
        for host in inventory['all']['hosts']:
            inventory['_meta']['hostvars'][host] = vars
    return inventory


def make_inventory(j, dc=None):
    inventory = {}
    inventory['followers'] = {}
    inventory['leader'] = {}
    inventory['all'] = {}
    # create hosts subkey
    for key in inventory:
        inventory[key]['hosts'] = []

    if 'Servers' in j:
        for server in j['Servers']:
            address = server['Address'].split(':')[0]
            if server.get('Leader') is True:
                inventory['leader']['hosts'].append(address)
            else:
                inventory['followers']['hosts'].append(address)
            inventory['all']['hosts'].append(address)
    return inventory


def backup_inventory(inventory, dc):
    with safe_open("{}/{}.json".format(INVENTORY_BACKUP_DIR, dc), 'w') as f:
        f.write(inventory)


if __name__ == '__main__':
    parser = argparse.ArgumentParser(description='Ansible inventory script')
    parser.add_argument('--list', action='store_true')
    args, unknown_args = parser.parse_known_args()

    dc = os.getenv('DATACENTER')

    if len(unknown_args) > 0:
        dc = unknown_args[0]

    if dc in [None, ""]:
        raise ValueError("""dc is None or empty string. A valid dc must be provided via DATACENTER environment variable
            or as arguments""")

    if is_reachable():
        servers = get_servers(dc)
        inventory_contents = json.dumps(insert_hostvars(make_inventory(servers, dc), get_failure_tolerance(dc)))
        backup_inventory(inventory_contents, dc)
    else:
        with safe_open("{}/{}.json".format(INVENTORY_BACKUP_DIR, dc), 'r') as f:
            inventory_contents = json.dumps(json.load(f))
    print(inventory_contents)
