package main

import (
	"context"
	"errors"
	"fmt"
	"log"
	"os"
	"strings"
	"sync"
	"time"

	"github.com/aws/aws-lambda-go/events"
	"github.com/aws/aws-lambda-go/lambda"
	"github.com/hashicorp/consul/api"
	"github.com/myENA/consul-backinator/command/backup"
)

const (
	BucketEnv       = "bucket_name"  // Env var for Bucket to use as output
	ConsulAddrEnv   = "consul_addr"  // Env var for consul endpoint address
	ConsulSchemeEnv = "consul_https" // Env var for consul https
)

var (
	S3Bucket     string
	ConsulAddr   string
	ConsulScheme string
)

var (
	MissingBucketErr     = errors.New(fmt.Sprintf("Missing S3 Bucket name (env: \"%s\")", BucketEnv))
	MissingConsulAddrErr = errors.New(fmt.Sprintf("Missing Consul Addr (env: \"%s\")", ConsulAddrEnv))
)

func init() {
	log.SetPrefix("")
	log.SetFlags(0)
	S3Bucket = os.Getenv(BucketEnv)
	ConsulAddr = os.Getenv(ConsulAddrEnv)

	ConsulScheme = "https" // default scheme
	if strings.ToLower(os.Getenv(ConsulSchemeEnv)) == "false" {
		ConsulScheme = "http"
	}
}

// Create a new default consul client with hardcoded defaults
func newClient() (*api.Client, error) {
	config := api.DefaultNonPooledConfig()
	config.Address = ConsulAddr
	config.Scheme = ConsulScheme
	return api.NewClient(config)
}

// Returns all datacenters from consul
func GetDatacenters() ([]string, error) {
	client, err := newClient()
	if err != nil {
		return nil, err
	}

	return client.Catalog().Datacenters()
}

// creates a new backinator backup.Command with no frills logging and prefix set to datacenter
func newBackup(datacenter string) *backup.Command {
	return &backup.Command{
		Self: os.Args[0], // use the name of the binary
		Log:  log.New(os.Stderr, fmt.Sprintf("%s ", datacenter), 0),
	}
}

// Backup a single datacenter KV
func BackupDatacenter(datacenter string) error {
	backup := newBackup(datacenter)

	// args need to be in CLI form because we're wrapping a CLI tool
	// and running this interface directly https://godoc.org/github.com/mitchellh/cli#Command
	// -key argument is missing, and will use consul-backinators default passphrase
	args := []string{
		fmt.Sprintf("-addr=%s", ConsulAddr),
		fmt.Sprintf("-scheme=%s", ConsulScheme),
		fmt.Sprintf("-dc=%s", datacenter),
		fmt.Sprintf("-file=s3://%s/%s/consul.bak", S3Bucket, datacenter),
	}
	if err := backup.Run(args); err > 0 {
		// backup.Run is an implementation of the Command interface from mitchellh/cli
		// need to convert integer error code to real error
		return errors.New(fmt.Sprintf("Error executing backup for %s.", datacenter))
	}
	return nil
}

func worker(dc string, wg *sync.WaitGroup, errCh chan error) {
	// worker function, triggers a backup of given Datacenter
	defer wg.Done()

	if err := BackupDatacenter(dc); err != nil {
		errCh <- err
	}
	return
}

func HandleRequest(ctx context.Context, event events.CloudWatchEvent) error {
	// validate we have strings from environment
	if S3Bucket == "" {
		return MissingBucketErr
	}

	if ConsulAddr == "" {
		return MissingConsulAddrErr
	}

	// get all datacenters
	datacenters, err := GetDatacenters()
	if err != nil {
		return err
	}

	var wg sync.WaitGroup
	errCh := make(chan error, len(datacenters))

	// run the backup command for each Datacenter
	// this creates a goroutine per datacenter and runs them all in parallel
	for _, dc := range datacenters {
		wg.Add(1)
		time.Sleep(time.Duration(1) * time.Second)
		go worker(dc, &wg, errCh)
	}

	// Wait for all goroutines to call wg.Done()
	wg.Wait()

	// Close the channel
	close(errCh)

	for err := range errCh {
		log.Print(err.Error())
	}

	return nil
}

func main() {
	lambda.Start(HandleRequest)
}
