package main

import (
	"errors"
	"flag"
	"fmt"
	"os"
	"reflect"
	"strconv"
	"strings"
	"unicode"
	"unicode/utf8"

	"code.justin.tv/edge/go-statsd-proxy/proxy"
)

// Configuration holds config information about how go-statsd-proxy should run
type configuration struct {
	Addresses      []string `doc:"statsd 'host:port' addresses to forward metrics to"`
	BufferBytes    int      `doc:"size of individual buffers in bytes"`
	BufferPoolSize int      `doc:"number of buffers in the collector buffer pool"`
	Hash           string   `doc:"hash to load balance, either 'fnv1a' or 'sha1'"`
	HealthPort     int      `doc:"port to listen for HTTP health check requests"`
	MaxPacketBytes int      `doc:"maximum packet size to emit from go-statsd-proxy"`
	NumWorkers     int      `doc:"number of udp forwarder workers"`
	StatsPort      int      `doc:"port to listen for statsd packets on"`

	// autoconfig using cluster+service
	UpstreamCluster string `doc:"Upstream cluster for ECS auto-configuration"`
	UpstreamService string `doc:"Upstream service for ECS auto-configuration"`
}

// Parse the command-line args to get the config filename then merge the
// provided (usually default) configuration with the configuration file.
func parse(args []string) (configuration, error) {

	// Set the default configuration options
	c := configuration{
		BufferBytes:    64 * 1024, // 64KB
		BufferPoolSize: 200,
		Hash:           proxy.FNV1A,
		HealthPort:     8080,
		MaxPacketBytes: 1420,
		NumWorkers:     1,
		StatsPort:      8125,
	}

	// Read options from the environment
	setFromEnv(&c)

	// Read options from the command line
	f := configuration{}
	if err := setFromFlags(args, &f); err != nil {
		return c, err
	}

	// Merge options, with flag arguments taking precedence over environment arguments.
	return merge(f, c), nil
}

// setFromFlags using reflection to place the correct types of values onto the parameters.
func setFromFlags(args []string, c *configuration) error {
	fs := flag.NewFlagSet(args[0], flag.ExitOnError)

	v := reflect.ValueOf(c).Elem()
	t := reflect.TypeOf(c).Elem()

	for i := 0; i < t.NumField(); i++ {
		f := v.Field(i)
		ft := t.Field(i)

		if !f.CanSet() {
			continue
		}

		// Make the name kebab-case for the flags.
		name := strings.ToLower(strings.Join(split(ft.Name), "-"))
		fs.Var(&value{f}, name, ft.Tag.Get("doc"))
	}

	if err := fs.Parse(args[1:]); err != nil {
		return err
	}

	return nil
}

// This type implements flag.Value with a reflect.Value,
// so we don't have to manually define flags.
type value struct {
	reflect.Value
}

// String is used by FlagSet to return a default value.
// TODO: make this return something more meaningful?
func (v *value) String() string {
	return v.Value.String()
}

// Set implements flag.Value.
func (v *value) Set(s string) error {
	switch v.Type().Kind() {
	case reflect.String:
		v.SetString(s)
	case reflect.Int:
		i, err := strconv.ParseInt(s, 10, 64)
		if err != nil {
			return err
		}
		v.SetInt(i)
	case reflect.Slice:
		values := getValues(s)
		sl := reflect.MakeSlice(v.Type(), len(values), len(values))

		for i, v := range values {
			sl.Index(i).SetString(v)
		}
		v.Value.Set(sl)
	}
	return nil
}

// setFromEnv variables, onto the provided configuration using reflection.
func setFromEnv(c *configuration) {
	t := reflect.TypeOf(c).Elem()
	v := reflect.ValueOf(c).Elem()

	for i := 0; i < t.NumField(); i++ {
		f := v.Field(i)
		ft := t.Field(i)
		typ := ft.Type

		// Skip struct fields we can't set (i.e. unexported fields).
		if !f.CanSet() {
			continue
		}

		// Transform the field name from CamelCase to SCARY_SNAKE case.
		name := strings.ToUpper(strings.Join(split(ft.Name), "_"))

		value := os.Getenv(name)
		if value == "" {
			continue
		}

		switch typ.Kind() {
		// Handle string fields
		case reflect.String:
			f.SetString(value)

		// Handle int fields
		case reflect.Int:
			if x, err := strconv.ParseInt(value, 10, 64); err == nil && x > 0 {
				f.SetInt(x)
			}

		// Assume we're reflecting into a []string.
		case reflect.Slice:
			// Parse comma-separated values, if we have them.
			values := getValues(value)
			sl := reflect.MakeSlice(typ, len(values), len(values))

			for i, v := range values {
				sl.Index(i).SetString(v)
			}

			f.Set(sl)

		default:
			continue
		}
	}
}

// merge a into b, with non-zero values in a taking precedence over values in b.
func merge(a, b configuration) configuration {
	av := reflect.ValueOf(a)
	bv := reflect.ValueOf(b)

	for i := 0; i < av.Type().NumField(); i++ {
		af := av.Field(i)
		bf := bv.Field(i)

		switch af.Type().Kind() {
		case reflect.Int:
			if !af.IsZero() {
				bf.SetInt(af.Int())
			}
		case reflect.String:
			if !af.IsZero() {
				bf.SetString(af.String())
			}
		case reflect.Slice:
			if af.Len() > 0 {
				bf.Set(af)
			}
		}
	}

	return b
}

// getValues from a comma-separated string.
func getValues(csv string) []string {
	if csv == "" {
		return nil
	}

	values := strings.Split(csv, ",")

	for i, v := range values {
		v = strings.TrimSpace(v)
		if x, err := strconv.Unquote(v); err == nil {
			v = x
		}
		values[i] = v
	}

	return values
}

func validate(c configuration) error {
	if c.Hash != proxy.SHA1 && c.Hash != proxy.FNV1A {
		return fmt.Errorf(`invalid hash function %q (only "sha1" and "fnv1a" are supported)`, c.Hash)
	}

	if (c.UpstreamCluster != "") != (c.UpstreamService != "") {
		return errors.New("Upstream cluster and upstream service must both be supplied")
	}

	if len(c.Addresses) == 0 && c.UpstreamCluster == "" {
		return errors.New("you must specify addresses or upstream configuration")
	}

	return nil
}

// Split splits the camelcase word and returns a list of words. It also
// supports digits. Both lower camel case and upper camel case are supported.
// For more info please check: http://en.wikipedia.org/wiki/CamelCase
//
// Examples
//
//   "" =>                     [""]
//   "lowercase" =>            ["lowercase"]
//   "Class" =>                ["Class"]
//   "MyClass" =>              ["My", "Class"]
//   "MyC" =>                  ["My", "C"]
//   "HTML" =>                 ["HTML"]
//   "PDFLoader" =>            ["PDF", "Loader"]
//   "AString" =>              ["A", "String"]
//   "SimpleXMLParser" =>      ["Simple", "XML", "Parser"]
//   "vimRPCPlugin" =>         ["vim", "RPC", "Plugin"]
//   "GL11Version" =>          ["GL", "11", "Version"]
//   "99Bottles" =>            ["99", "Bottles"]
//   "May5" =>                 ["May", "5"]
//   "BFG9000" =>              ["BFG", "9000"]
//   "BöseÜberraschung" =>     ["Böse", "Überraschung"]
//   "Two  spaces" =>          ["Two", "  ", "spaces"]
//   "BadUTF8\xe2\xe2\xa1" =>  ["BadUTF8\xe2\xe2\xa1"]
//
// Splitting rules
//
//  1) If string is not valid UTF-8, return it without splitting as
//     single item array.
//  2) Assign all unicode characters into one of 4 sets: lower case
//     letters, upper case letters, numbers, and all other characters.
//  3) Iterate through characters of string, introducing splits
//     between adjacent characters that belong to different sets.
//  4) Iterate through array of split strings, and if a given string
//     is upper case:
//       if subsequent string is lower case:
//         move last character of upper case string to beginning of
//         lower case string
//
// Credit to Fatih Arslan:
// https://github.com/fatih/camelcase/blob/9db1b65eb38bb28986b93b521af1b7891ee1b04d/camelcase.go
func split(s string) (entries []string) {
	// don't split invalid utf8
	if !utf8.ValidString(s) {
		return []string{s}
	}
	entries = []string{}
	var runes [][]rune
	lastClass := 0
	class := 0
	// split into fields based on class of unicode character
	for _, r := range s {
		switch true {
		case unicode.IsLower(r):
			class = 1
		case unicode.IsUpper(r):
			class = 2
		case unicode.IsDigit(r):
			class = 3
		default:
			class = 4
		}
		if class == lastClass {
			runes[len(runes)-1] = append(runes[len(runes)-1], r)
		} else {
			runes = append(runes, []rune{r})
		}
		lastClass = class
	}
	// handle upper case -> lower case sequences, e.g.
	// "PDFL", "oader" -> "PDF", "Loader"
	for i := 0; i < len(runes)-1; i++ {
		if unicode.IsUpper(runes[i][0]) && unicode.IsLower(runes[i+1][0]) {
			runes[i+1] = append([]rune{runes[i][len(runes[i])-1]}, runes[i+1]...)
			runes[i] = runes[i][:len(runes[i])-1]
		}
	}
	// construct []string from results
	for _, s := range runes {
		if len(s) > 0 {
			entries = append(entries, string(s))
		}
	}
	return
}
