// Package hashring implements discretized consistent hashing.
//
// https://medium.com/system-design-blog/consistent-hashing-b9134c8a9062
//
// This creates a fairly entropic hashring filling in a circle, but to avoid comparisons or b-trees,
// we tradeoff using a tiny bit of extra memory (524kb) and discretize the circle into 65,536 buckets so that
// an integer modulo lookup is sufficient, allowing O(1) lookup instead of O(log n) lookup.
package hashring

import (
	"fmt"
	"hash/fnv"
	"sort"

	"code.justin.tv/edge/go-statsd-proxy/proxy"
)

const ringSize = 65536

func MakeRing(forwarders []proxy.Forwarder, labels []string, replicas int) ([]proxy.Forwarder, error) {
	if len(forwarders) != len(labels) {
		return nil, fmt.Errorf("Creating hash ring len(forwarders) != len(labels); %d != %d", len(forwarders), len(labels))
	}

	// Phase 1: sort all the forwarders  so that DNS roundrobin doesn't make this non-deterministic.
	pairings := make([]pair, len(forwarders))
	for i := 0; i < len(forwarders); i++ {
		pairings[i] = pair{labels[i], forwarders[i]}
	}
	sort.Slice(pairings, func(i, j int) bool { return pairings[i].label < pairings[j].label })

	// Phase 2: sparsely fill the ring
	output := make([]proxy.Forwarder, ringSize)

	var lastForwarder proxy.Forwarder
	var biggestIndex uint32
	for _, p := range pairings {
		for i := 0; i < replicas; i++ {
			h := fnv.New32a()
			_, err := fmt.Fprintf(h, "%s_%d", p.label, i)
			if err != nil {
				return nil, fmt.Errorf("unexpected error in Fprintf: %s", err.Error())
			}
			idx := h.Sum32() % ringSize
			if idx >= biggestIndex {
				biggestIndex, lastForwarder = idx, p.forwarder
			}
			output[idx] = p.forwarder
		}
	}

	// Phase 3: fill in the gaps in the ring
	for i := 0; i < ringSize; i++ {
		if f := output[i]; f == nil {
			output[i] = lastForwarder
		} else {
			lastForwarder = f
		}
	}

	return output, nil
}

type pair struct {
	label     string
	forwarder proxy.Forwarder
}
