package ecs_test

import (
	"errors"
	"io/ioutil"
	"net/http"
	"net/http/httptest"
	"path"
	"testing"

	"github.com/aws/aws-sdk-go/aws/ec2metadata"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"

	"code.justin.tv/edge/go-statsd-proxy/internal/ecs"
)

func TestGetMetadata(t *testing.T) {
	// This is the response we're looking for.
	expected := &ecs.Metadata{
		InstanceID: "i-1234567890abcdef0",
		TaskID:     "febee046097849aba589d4435207c04a",
		Region:     "us-west-2",
	}

	// Setup mocks for the Task Metadata endpoint and ec2metadata function.
	s := httptest.NewServer(mockTaskMetadataResponse(t))
	ec2 := mockGetter{instanceID: expected.InstanceID, region: expected.Region}

	meta, err := ecs.GetMetadata(s.URL, ec2)

	require.NoError(t, err)
	require.Equal(t, expected, meta)
}

type mockGetter struct {
	instanceID string
	region     string
}

// Implements the ecs.Getter interface.
func (g mockGetter) GetInstanceIdentityDocument() (ec2metadata.EC2InstanceIdentityDocument, error) {
	// Provide an easy way to return a mocked error; leave values unset.
	if g.instanceID == "" || g.region == "" {
		return ec2metadata.EC2InstanceIdentityDocument{}, errors.New("mock error")
	}

	// Otherwise return mocked values when unset.
	return ec2metadata.EC2InstanceIdentityDocument{
		InstanceID: g.instanceID,
		Region:     g.region,
	}, nil
}

func mockTaskMetadataResponse(t *testing.T) http.HandlerFunc {
	t.Helper()

	return func(w http.ResponseWriter, r *http.Request) {
		b, err := ioutil.ReadFile(path.Join("testdata", "task_metadata_v4.json"))
		require.NoError(t, err)

		_, err = w.Write(b)
		assert.NoError(t, err)
	}
}
