package fingerprint

import (
	"encoding/hex"
	"fmt"
	"hash/fnv"
	"net"
	"strings"
)

func Fingerprint(hosts []string) string {
	h := fnv.New64a()
	for _, host := range hosts {
		_, _ = h.Write([]byte(host))
	}
	return hex.EncodeToString(h.Sum(nil))
}

func Summary(hosts []string, includeHosts bool) string {
	fingerprint := Fingerprint(hosts)
	summary := fmt.Sprintf("%d hosts (fp %s)", len(hosts), fingerprint)
	if includeHosts {
		hostPort := make(map[string][]string)

		for _, h := range hosts {
			host, port, err := net.SplitHostPort(h)
			if err == nil {
				hostPort[host] = append(hostPort[host], port)
			}
		}
		var output []string
		for host, ports := range hostPort {
			if len(ports) == 1 {
				output = append(output, host+":"+ports[0])
			} else {
				output = append(output, fmt.Sprintf("%s:{%s}", host, strings.Join(ports, ",")))
			}
		}
		summary += ": " + strings.Join(output, ", ")
	}
	return summary
}
