package hostconfig

import (
	"context"
	"errors"
	"fmt"
	"sort"

	"code.justin.tv/edge/go-statsd-proxy/internal/observe"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/ec2"
	"github.com/aws/aws-sdk-go/service/ecs"
)

type ServiceConfig struct {
	ServiceName string
	ClusterName string
}

var ErrNoTasks = errors.New("No tasks matched")

// getStatsHosts finds all tasks running with the provided task definition, and returns a list of host port strings for each task of form "{host_private_ip}:{host_port_for_task}".
func GetStatsHosts(ctx context.Context, obs observe.Observer, ecsClient *ecs.ECS, ec2Client *ec2.EC2, conf *ServiceConfig) ([]string, error) {
	allTasks, err := getTasks(ctx, ecsClient, conf)
	if err != nil {
		return nil, err
	}

	// Filter to tasks that are running the specified task definition.
	tasks := make([]*ecs.Task, 0, len(allTasks))
	for _, t := range allTasks {
		status := aws.StringValue(t.LastStatus)
		health := aws.StringValue(t.HealthStatus)
		if status == ecs.DesiredStatusRunning && health != ecs.HealthStatusUnhealthy {
			tasks = append(tasks, t)
		} else {
			obs.Debug("Ignored task", "task", aws.StringValue(t.TaskArn), "status", status, "health", health)
		}
	}

	if len(tasks) == 0 {
		return nil, ErrNoTasks
	}

	containerInstanceIPs, err := getContainerInstanceIPs(ctx, ecsClient, ec2Client, conf)
	if err != nil {
		return nil, err
	}

	statsiteHosts := make([]string, len(tasks))
	for i, task := range tasks {
		if len(task.Containers) > 1 {
			return nil, errors.New("only expected one container per statsite task")
		}

		containerIP, ok := containerInstanceIPs[aws.StringValue(task.ContainerInstanceArn)]
		if !ok {
			return nil, fmt.Errorf("container instance %s not found in statsite cluster", aws.StringValue(task.ContainerInstanceArn))
		}
		networkBindings := task.Containers[0].NetworkBindings
		for _, binding := range networkBindings {
			// StatsdProxy sends via udp, even though tcp is also available.
			if aws.StringValue(binding.Protocol) == "udp" {
				fullHost := fmt.Sprintf("%s:%d", containerIP, int(aws.Int64Value(binding.HostPort)))
				statsiteHosts[i] = fullHost
			}
		}
	}
	sort.Strings(statsiteHosts)
	return statsiteHosts, nil
}

// Returns all tasks currently running on the provided service and cluster. Handles pagination.
func getTasks(ctx context.Context, ecsClient *ecs.ECS, conf *ServiceConfig) ([]*ecs.Task, error) {
	fetchPage := func(nextToken *string) ([]*ecs.Task, *string, error) {
		listTasksOut, err := ecsClient.ListTasksWithContext(ctx, &ecs.ListTasksInput{
			Cluster:     &conf.ClusterName,
			ServiceName: &conf.ServiceName,
		})
		if err != nil {
			return nil, nil, err
		}

		out, err := ecsClient.DescribeTasksWithContext(ctx, &ecs.DescribeTasksInput{
			Cluster: &conf.ClusterName,
			Tasks:   listTasksOut.TaskArns,
		})
		if err != nil {
			return nil, nil, err
		}

		return out.Tasks, listTasksOut.NextToken, nil
	}

	tasks, nextToken, err := fetchPage(nil)
	if err != nil {
		return nil, err
	}

	var t []*ecs.Task
	for nextToken != nil {
		t, nextToken, err = fetchPage(nextToken)
		if err != nil {
			return nil, err
		}

		tasks = append(tasks, t...)
	}

	return tasks, nil
}

// getContainerInstanceIPs returns a map of containerInstanceARN to host private IP.
func getContainerInstanceIPs(ctx context.Context, ecsClient *ecs.ECS, ec2Client *ec2.EC2, conf *ServiceConfig) (map[string]string, error) {
	// Describe all container instances.
	listIn := &ecs.ListContainerInstancesInput{
		Cluster: &conf.ClusterName,
	}
	listOut, err := ecsClient.ListContainerInstancesWithContext(ctx, listIn)
	if err != nil {
		return nil, err
	}
	describeIn := &ecs.DescribeContainerInstancesInput{
		Cluster:            &conf.ClusterName,
		ContainerInstances: listOut.ContainerInstanceArns,
	}
	describeOut, err := ecsClient.DescribeContainerInstancesWithContext(ctx, describeIn)
	if err != nil {
		return nil, err
	}

	// Builds list of instanceIDs to fetch, and map of instanceID to container instance ARN.
	instanceIDs := make([]string, len(describeOut.ContainerInstances))
	instanceIDToContainerDefArn := make(map[string]string)
	for i, containerInstance := range describeOut.ContainerInstances {
		instanceID := aws.StringValue(containerInstance.Ec2InstanceId)
		instanceIDs[i] = instanceID
		instanceIDToContainerDefArn[instanceID] = aws.StringValue(containerInstance.ContainerInstanceArn)
	}

	describeInstancesIn := &ec2.DescribeInstancesInput{
		InstanceIds: aws.StringSlice(instanceIDs),
	}
	describeInstancesOut, err := ec2Client.DescribeInstancesWithContext(ctx, describeInstancesIn)
	if err != nil {
		return nil, err
	}

	// Build final map of containerInstanceARN to host private IP.
	containerInstanceArnToIP := make(map[string]string)
	for _, reservation := range describeInstancesOut.Reservations {
		for _, instance := range reservation.Instances {
			containerInstanceArn := instanceIDToContainerDefArn[aws.StringValue(instance.InstanceId)]
			containerInstanceArnToIP[containerInstanceArn] = aws.StringValue(instance.PrivateIpAddress)
		}
	}

	return containerInstanceArnToIP, nil
}
