package hostconfig

import (
	"context"
	"time"

	"code.justin.tv/edge/go-statsd-proxy/internal/fingerprint"
	"code.justin.tv/edge/go-statsd-proxy/internal/observe"
	"github.com/aws/aws-sdk-go/service/ec2"
	"github.com/aws/aws-sdk-go/service/ecs"
)

const minPollInterval = 5 * time.Second
const maxPollInterval = 90 * time.Second
const incrementPoll = 100 * time.Millisecond

const requestTimeout = 20 * time.Second

type ReconfigureFunc func(hosts []string)

type Poller struct {
	Observer observe.Observer
	ECS      *ecs.ECS
	EC2      *ec2.EC2
	Conf     *ServiceConfig
}

func (p *Poller) Poll(baseCtx context.Context, reconfigure ReconfigureFunc) {
	getHostsTimeout := func() ([]string, error) {
		ctx, cancel := context.WithTimeout(baseCtx, requestTimeout)
		defer cancel()
		return GetStatsHosts(ctx, p.Observer, p.ECS, p.EC2, p.Conf)
	}

	hosts, err := getHostsTimeout()
	if err != nil {
		p.Observer.Warn("Could not poll stats hosts", "err", err)
		return
	}

	pollInterval := minPollInterval
	var candidateRepeats int
	var candidate []string

	for {
		select {
		case <-baseCtx.Done():
			return
		case <-time.After(pollInterval):
		}
		if pollInterval < maxPollInterval {
			pollInterval += incrementPoll
		}

		curHosts, err := getHostsTimeout()

		if err != nil {
			// We're okay with errors in the poller due to temporary unavailability, etc
			p.Observer.Info("Could not poll stats hosts", "err", err)
			continue
		}
		if !stringSliceEqual(curHosts, hosts) {
			pollInterval = minPollInterval
			if !stringSliceEqual(curHosts, candidate) {
				candidate = curHosts
				candidateRepeats = 0
			}
			candidateRepeats += 1

			// During the deploy the hosts list will change many times as more instances come up.
			// We want the hosts to "settle" by repeating the same hosts in three intervals.
			if candidateRepeats >= 3 {
				p.Observer.Debug(
					"reconfiguring hosts",
					"old", fingerprint.Summary(hosts, true),
					"new", fingerprint.Summary(curHosts, true),
				)
				go reconfigure(curHosts)
				hosts = curHosts
				candidate, candidateRepeats = nil, 0
			} else {
				p.Observer.Debug(
					"waiting for hosts to settle",
					"candidate", fingerprint.Summary(candidate, false),
					"seen", candidateRepeats,
				)
			}
		}
	}
}

func stringSliceEqual(a, b []string) bool {
	if len(a) != len(b) {
		return false
	}
	for i, v := range a {
		if v != b[i] {
			return false
		}
	}
	return true
}
