package observe

import (
	identifier "code.justin.tv/amzn/TwitchProcessIdentifier"
	"code.justin.tv/edge/go-statsd-proxy/internal/build"

	"go.uber.org/zap"
	"go.uber.org/zap/zapcore"
)

// Logger is a structured logger.
type Logger struct {
	*zap.Logger
}

// NewLogger creates a new Logger instance which outputs to the 'output' path.
// Each log line will include the provided build context as well.
func NewLogger(output string, ctx build.Context) (*Logger, error) {
	config := zap.Config{
		Level:             zap.NewAtomicLevelAt(zap.DebugLevel),
		DisableCaller:     false,
		DisableStacktrace: false,
		Encoding:          "json",
		EncoderConfig: zapcore.EncoderConfig{
			LevelKey:       "level",
			MessageKey:     "msg",
			NameKey:        "logger",
			StacktraceKey:  "trace",
			TimeKey:        "ts",
			EncodeCaller:   zapcore.ShortCallerEncoder,
			EncodeDuration: zapcore.MillisDurationEncoder,
			EncodeLevel:    zapcore.LowercaseLevelEncoder,
			EncodeTime:     zapcore.RFC3339NanoTimeEncoder,
			LineEnding:     zapcore.DefaultLineEnding,
		},
		OutputPaths:      []string{output},
		ErrorOutputPaths: []string{output},
		Sampling: &zap.SamplingConfig{
			Initial:    100,
			Thereafter: 100,
		},
	}

	// Add these fields to any log message.
	withFields := zap.Fields(
		buildContext(ctx),
	)

	// Build a logger with the above configuration and standard fields.
	log, err := config.Build(withFields)
	if err != nil {
		return nil, err
	}

	return &Logger{Logger: log}, nil
}

// AddProcessID adds the process identifier field to the logger.
// Not safe for concurrent use.
func (l *Logger) AddProcessID(pid identifier.ProcessIdentifier) {
	l.Logger = l.Logger.With(processID(pid))
	_ = zap.ReplaceGlobals(l.Logger)
}

func (l *Logger) Error(msg string, err error, kv ...interface{}) {
	fields := make([]zap.Field, 0, (len(kv)/2)+1)
	fields = append(fields, zap.Error(err))
	fields = append(fields, kvFields(kv)...)
	l.Logger.Error(msg, fields...)
}

func (l *Logger) Info(msg string, kv ...interface{}) {
	l.Logger.Info(msg, kvFields(kv)...)
}
func (l *Logger) Debug(msg string, kv ...interface{}) {
	l.Logger.Debug(msg, kvFields(kv)...)
}
func (l *Logger) Warn(msg string, kv ...interface{}) {
	l.Logger.Warn(msg, kvFields(kv)...)
}

func kvFields(kv []interface{}) []zap.Field {
	fields := make([]zap.Field, 0, len(kv)/2)

	for i := 0; i+1 < len(kv); i += 2 {
		key, ok := kv[i].(string)
		if !ok {
			continue
		}

		fields = append(fields, zap.Any(key, kv[i+1]))
	}
	return fields
}

// Log implements the MetricsLogger interface.
func (l *Logger) Log(msg string, kv ...interface{}) {
	l.Info(msg, kv...)
}

type NoopLogger struct{}

func (NoopLogger) Error(string, error, ...interface{}) {}
func (NoopLogger) Info(string, ...interface{})         {}
func (NoopLogger) Log(string, ...interface{})          {}
func (NoopLogger) Debug(string, ...interface{})        {}
func (NoopLogger) Warn(string, ...interface{})         {}
