package observe

import (
	"time"

	"github.com/aws/aws-sdk-go/aws/session"

	id "code.justin.tv/amzn/TwitchProcessIdentifier"
	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	cw "code.justin.tv/amzn/TwitchTelemetryCloudWatchMetricsSender"
)

type MetricsLogger interface {
	Log(msg string, kv ...interface{})
}

type Metrics struct {
	observer   telemetry.SampleObserver
	dimensions telemetry.DimensionSet
	rollups    [][]string
}

func NewMetrics(pid id.ProcessIdentifier, sess *session.Session, log MetricsLogger) (*Metrics, error) {
	sender := cw.NewUnbufferedWithSession(&pid, log, sess)

	flushPeriod := 30 * time.Second // Send metrics off to CloudWatch on this interval.
	bufferBytes := 5000
	bucketSize := time.Second // Aggregate any metrics into buckets of this size.

	observer := telemetry.NewBufferedAggregator(flushPeriod, bufferBytes, bucketSize, sender, log)

	dimensions := map[string]string{
		"Region": pid.Region,
		"TaskID": pid.LaunchID,
	}

	rollups := [][]string{
		{"TaskID"},
		{"Region", "TaskID"},
	}

	return &Metrics{
		observer:   observer,
		dimensions: dimensions,
		rollups:    rollups,
	}, nil
}

// Increment the metric by the provided value.
func (m *Metrics) Increment(metric string, value int64) {
	sample := &telemetry.Sample{
		MetricID: telemetry.MetricID{
			Name:       metric,
			Dimensions: m.dimensions,
		},
		RollupDimensions: m.rollups,
		Timestamp:        time.Now(),
		Value:            float64(value),
		Unit:             "Count",
	}

	m.observer.ObserveSample(sample)
}

// Time records the difference between the provided t and when this method is executed.
func (m *Metrics) Time(metric string, t time.Time) {
	// Call time.Now to get the current time, then use to result to calculate the difference
	// instead of calling time.Since.
	//
	// This halves the amount of calls to get the current time.
	now := time.Now()
	durationMillis := now.Sub(t).Milliseconds()

	sample := &telemetry.Sample{
		MetricID: telemetry.MetricID{
			Name:       metric,
			Dimensions: m.dimensions,
		},
		RollupDimensions: m.rollups,
		Timestamp:        now,
		Value:            float64(durationMillis),
		Unit:             "Milliseconds",
	}

	m.observer.ObserveSample(sample)
}

func (m *Metrics) Close() {
	m.observer.Flush()
	m.observer.Stop()
}

type NoopMetrics struct{}

func (NoopMetrics) Increment(string, int64) {}
func (NoopMetrics) Time(string, time.Time)  {}
