package proxy

import (
	"net"
	"os"
	"runtime"
	"sync"
	"time"
)

// TODO: Make some shared BufferPool struct that all collectors use

// Collector gets packet data from a source and sends it into a channel
type Collector interface {
	Run()
	GetBuffer() []byte
	ReturnBuffer(buf []byte)
}

// UDPCollector collects packets from UDP
type UDPCollector struct {
	BufferSize int
	BufferPool chan []byte
	DataPool   chan []byte
	conn       *net.UDPConn
	observer   Observer
}

const udpBufferSize = 5242880 // 5MiB

// NewUDPCollector makes a UDPCollector
func NewUDPCollector(port, poolSize, bufferBytes int, o Observer) (*UDPCollector, error) {
	bufferPool := make(chan []byte, poolSize) // holds buffers ready to get network data
	dataPool := make(chan []byte, poolSize)   // holds buffers with data ready to be consumed downstream
	// Fill the empty bufferPool of re-usable buffers with fresh buffers
	for i := 0; i < poolSize; i++ {
		buf := make([]byte, bufferBytes)[:0]
		bufferPool <- buf
	}

	addr := &net.UDPAddr{
		Port: port,
		IP:   net.IPv4(0, 0, 0, 0),
	}

	conn, err := net.ListenUDP("udp", addr)
	if err != nil {
		return nil, err
	}

	// increase the size of the recv buffer to smooth out metric submission spikes
	err = conn.SetReadBuffer(udpBufferSize)
	if err != nil {
		return nil, err
	}

	return &UDPCollector{
		BufferSize: bufferBytes,
		BufferPool: bufferPool,
		DataPool:   dataPool,
		conn:       conn,
		observer:   o,
	}, nil
}

// Run run
func (uc *UDPCollector) Run() {
	// Listen on the UDP Address
	runtime.LockOSThread()
	defer uc.conn.Close()

	// Fire an event telling us to report stats every 500ms
	statTick := time.NewTicker(500 * time.Millisecond)
	defer statTick.Stop()

	// Timeout-generator ticker for flushing packets
	// Start it on some massive time so it doesn't accidentally fire on startup
	timeout := time.NewTicker(3 * time.Second)
	defer timeout.Stop()

	// all this goes inside the function where the double nested for loop is currently
	packetCountMutex := &sync.Mutex{}
	bufMutex := &sync.Mutex{}
	var packetInCount int64
	packetInCount = 0

	var err error

	// Buffer to fill with packets (popped from BufferPool)
	var buf []byte
	// Local buffer to read packets into
	var recvBuf = make([]byte, 65536)

	// Keep track of buffer lengths
	var recvBufLen int

	// Periodically flush statistics
	go func() {
		for {
			// Wait on the ticker until its time to submit stats
			<-statTick.C
			packetCountMutex.Lock()
			localPacketInCount := packetInCount
			packetInCount = 0
			packetCountMutex.Unlock()

			uc.observer.Increment("BufferPoolSize", int64(len(uc.BufferPool)))
			uc.observer.Increment("DataPoolSize", int64(len(uc.DataPool)))
			uc.observer.Increment("PacketsIn", localPacketInCount)
		}
	}()

	// Small function to encapsulate buffer flushing logic
	flush := func() {
		uc.DataPool <- buf
		buf = <-uc.BufferPool
		buf = buf[:0]
	}

	// Set a timeout for flushing buffers outside of the blocking UDP read section
	go func() {
		for {
			<-timeout.C
			bufMutex.Lock()
			// only flush if the buffer has something in it
			if len(buf) != 0 {
				flush() //affects buf
			}
			bufMutex.Unlock()
		}
	}()

	// Get initial buffer
	bufMutex.Lock()
	buf = <-uc.BufferPool
	bufMutex.Unlock()

	// Fill buffers forever!
	for {
		if recvBufLen, _, err = uc.conn.ReadFromUDP(recvBuf); err != nil {
			uc.observer.Error("unable to read from UDP", err)
			os.Exit(1) // TODO: do not kill the program from here.
		}

		// Grab lock to modify buf
		bufMutex.Lock()

		// if we can't fit this packet in this buffer, flush and get a new buffer first
		if len(buf)+recvBufLen+1 > uc.BufferSize {
			flush() // gets a new buf
		}
		// Add the current packet to the buffer
		if len(buf) != 0 {
			buf = append(buf, byte('\n')) // not first packet, add a newline
		}
		buf = append(buf, recvBuf[:recvBufLen]...)

		bufMutex.Unlock()

		packetCountMutex.Lock()
		packetInCount++
		packetCountMutex.Unlock()
	}
}

// GetBuffer gets buffer
func (uc *UDPCollector) GetBuffer() []byte {
	buf := <-uc.DataPool // Pulls a buffer from the data pool
	return buf
}

// ReturnBuffer returns buffer
func (uc *UDPCollector) ReturnBuffer(buf []byte) {
	buf = buf[:uc.BufferSize] // reset the size of the buffer to max size
	uc.BufferPool <- buf      // Return the buffer to the free pool
}
