package proxy

import (
	"net"
	"testing"
	"time"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestCollectorConstruction(t *testing.T) {
	uc, err := NewUDPCollector(8123, 500, 64*1024, NoopObserver{})
	require.NoError(t, err)

	assert.Equal(t, 500, len(uc.BufferPool))
	assert.Equal(t, 0, len(uc.DataPool))
	assert.Equal(t, 64*1024, uc.BufferSize)
}

func TestCollectorGetBuffer(t *testing.T) {
	// Use tiny buffers to ensure flushes happen
	uc, err := NewUDPCollector(8131, 100, 70, NoopObserver{})
	require.NoError(t, err)

	go uc.Run()
	// 1 buffer poppped off the BufferPool waiting to be written into
	// but no buffers ready to read by downstream
	time.Sleep(500 * time.Millisecond)
	assert.Equal(t, 99, len(uc.BufferPool))
	assert.Equal(t, 0, len(uc.DataPool))
	// Write some data in (len=56)
	p := []byte("some.metric:1|c\nanother.stat:45.0|ms|@0.1\nmore.data:55|g")
	conn, err := net.Dial("udp", "localhost:8131")
	require.NoError(t, err)

	_, err = conn.Write(p)
	require.NoError(t, err)

	// The first write will not trigger a buffer flush, because there is still
	// room in the buffer
	time.Sleep(500 * time.Millisecond)
	assert.Equal(t, 99, len(uc.BufferPool))
	assert.Equal(t, 0, len(uc.DataPool))

	_, err = conn.Write(p)
	require.NoError(t, err)

	// Now one buffer is ready to be sent downstream, one is waiting to be written
	// into, and the rest are in the BufferPool
	time.Sleep(500 * time.Millisecond)
	assert.Equal(t, 98, len(uc.BufferPool))
	assert.Equal(t, 1, len(uc.DataPool))
	// Grab a packet off the DataPool
	fetchedPacket := uc.GetBuffer()
	// Now the DataPool is empty again
	assert.Equal(t, p, fetchedPacket)
	assert.Equal(t, len(p), len(fetchedPacket))
	assert.Equal(t, 70, cap(fetchedPacket))
	assert.Equal(t, 98, len(uc.BufferPool))
	assert.Equal(t, 0, len(uc.DataPool))
}

func TestCollectorMultiPacketBuffer(t *testing.T) {
	// Buffer size big enough to fit 3 small packets per buffer
	uc, err := NewUDPCollector(8128, 100, 200, NoopObserver{})
	require.NoError(t, err)

	go uc.Run()
	// 1 buffer popped off the BufferPool waiting to be written into
	// but no buffers ready to read by downstream
	time.Sleep(500 * time.Millisecond)
	assert.Equal(t, 99, len(uc.BufferPool))
	assert.Equal(t, 0, len(uc.DataPool))
	// Write some data in (len=56)
	p := []byte("some.metric:1|c\nanother.stat:45.0|ms|@0.1\nmore.data:55|g")
	correctBuf := make([]byte, 200)
	correctBuf = correctBuf[:0]
	correctBuf = append(correctBuf, p...)
	correctBuf = append(correctBuf, byte('\n'))
	correctBuf = append(correctBuf, p...)
	correctBuf = append(correctBuf, byte('\n'))
	correctBuf = append(correctBuf, p...)

	conn, err := net.Dial("udp", "localhost:8128")
	require.NoError(t, err)

	// Submit 1 packet
	_, err = conn.Write(p)
	require.NoError(t, err)

	// Now one buffer is ready to be sent downstream, one is waiting to be written
	// into, and the rest are in the BufferPool
	time.Sleep(500 * time.Millisecond)
	assert.Equal(t, 99, len(uc.BufferPool))
	assert.Equal(t, 0, len(uc.DataPool))

	// Submit 2nd packet
	_, err = conn.Write(p)
	require.NoError(t, err)

	time.Sleep(500 * time.Millisecond)
	assert.Equal(t, 99, len(uc.BufferPool))
	assert.Equal(t, 0, len(uc.DataPool))

	// Submit 3rd packet, still room on buffer
	_, err = conn.Write(p)
	require.NoError(t, err)

	time.Sleep(500 * time.Millisecond)
	assert.Equal(t, 99, len(uc.BufferPool))
	assert.Equal(t, 0, len(uc.DataPool))

	// Submit 4th packet; should trip a flush, and give us a buffer
	// with the first 3 packets
	_, err = conn.Write(p)
	require.NoError(t, err)

	time.Sleep(500 * time.Millisecond)
	assert.Equal(t, 98, len(uc.BufferPool))
	assert.Equal(t, 1, len(uc.DataPool))
	// Grab a packet off the DataPool
	fetchedPacket := uc.GetBuffer()
	// Now the DataPool is empty again
	assert.Equal(t, correctBuf, fetchedPacket)
	assert.Equal(t, len(correctBuf), len(fetchedPacket))
	assert.Equal(t, 200, cap(fetchedPacket))
	assert.Equal(t, 98, len(uc.BufferPool))
	assert.Equal(t, 0, len(uc.DataPool))
}

func TestCollectorReturnBuffer(t *testing.T) {
	uc, err := NewUDPCollector(8126, 500, 75, NoopObserver{})
	require.NoError(t, err)

	go uc.Run()
	// 1 buffer popped off the BufferPool waiting to be written into
	// but no buffers ready to read by downstream.
	time.Sleep(500 * time.Millisecond)
	assert.Equal(t, 499, len(uc.BufferPool))
	assert.Equal(t, 0, len(uc.DataPool))
	// Write some data in
	p := []byte("some.metric:1|c\nanother.stat:45.0|ms|@0.1\nmore.data:55|g")
	conn, err := net.Dial("udp", "localhost:8126")
	require.NoError(t, err)

	_, err = conn.Write(p)
	require.NoError(t, err)

	_, err = conn.Write(p) // this Write will trip a flush of the first packet
	require.NoError(t, err)

	fetchedPacket := uc.GetBuffer()
	// ... now you would do some stuff with that packet data
	// and then return it
	uc.ReturnBuffer(fetchedPacket)
	time.Sleep(1 * time.Second)
	assert.Equal(t, 499, len(uc.BufferPool))
	assert.Equal(t, 0, len(uc.DataPool))
}
