package proxy

import (
	"errors"
	"net"
	"sync"
	"time"
)

var (
	// errPacketTooLarge returned from appendLine when the line cannot fit on the packet
	errPacketTooLarge = errors.New("packet_too_large")
)

// Forwarder takes metrics and ships them to the downstream statsd nodes
type Forwarder interface {
	Run()
	SendLine(*BufferLine)
	Stop()
}

// UDPForwarder receives lines, batches them, and hands them to the destination
type UDPForwarder struct {
	queue          chan *BufferLine
	conn           net.Conn
	packet         []byte
	maxPacketSize  int
	packetOutCount int64
	observer       Observer
	shutdownOnce   sync.Once
}

// NewUDPForwarder creates a forwarder that sends data to host;
// Sleeps for 1 second, reads data off the queue, and sends in chunks of 10 metrics
func NewUDPForwarder(host string, maxPacketSize int, o Observer) (*UDPForwarder, error) {
	conn, err := net.Dial("udp", host)
	if err != nil {
		return nil, err
	}

	return &UDPForwarder{
		queue:         make(chan *BufferLine, 50000), // arbitrarily large for now to avoid a locking slowdown
		conn:          conn,
		packet:        make([]byte, 0, maxPacketSize), // truncated packet
		maxPacketSize: maxPacketSize,
		observer:      o,
	}, nil
}

// Run Runs forever and sends packets to the statsd endpoint,
// coalescing in a reasonably efficient way
func (uf *UDPForwarder) Run() {
	timeout := time.NewTicker(500 * time.Millisecond)

	for {
		select {
		case <-timeout.C: // timeout trigger to ensure we keep sending things and never deadlock
			// only send if the packet isn't empty
			if len(uf.packet) != 0 {
				uf.sendPacket()
				uf.truncatePacket()
			}
			// Flush the packetOutCount while we are at it, and reset it
			uf.observer.Increment("PacketsOut", uf.packetOutCount)
			uf.packetOutCount = 0
		case bufferLine, ok := <-uf.queue: // ... or get a line
			if !ok {
				return
			}
			uf.processLine(bufferLine)
		}
	}
}

// Stop this forwarder. Only call if you're sure nobody is sending data to this forwarder anymore.
func (uf *UDPForwarder) Stop() {
	uf.shutdownOnce.Do(func() {
		close(uf.queue)
	})
}

func (uf *UDPForwarder) processLine(bl *BufferLine) {
	// Done with the BufferLine, so call Done on it to notify upstream
	// We will always handle the line, one way or another, so defer the Done call at the top.
	defer bl.Done()

	line := bl.Line
	// Make sure the line isn't bigger than the maxPacketSize by itself
	if len(line) > uf.maxPacketSize {
		uf.observer.Error("stat line larger than max packet size", errors.New(string(line)),
			"maxPacketSize", uf.maxPacketSize, "lineSize", len(line))
		uf.observer.Increment("LineLengthErrorBytes", int64(len(line)))
		// Done processing this bad line
		return
	}

	if err := uf.appendLine(line); err == errPacketTooLarge {
		uf.sendPacket()
		uf.truncatePacket() // Truncate the packet back to empty

		// Append again, after freeing packet space
		if err := uf.appendLine(line); err != nil {
			uf.observer.Increment("LineLengthErrorBytes", int64(len(line)))
			uf.observer.Error("stat line too large, but was not filtered by size filter", errors.New(string(line)))
		}
	}
}

// SendLine submits a line for the forwarder to send
func (uf *UDPForwarder) SendLine(bufferLine *BufferLine) {
	uf.queue <- bufferLine
}

func (uf *UDPForwarder) sendPacket() {
	_, err := uf.conn.Write(uf.packet)
	// Best effort packet sending
	if err != nil {
		uf.observer.Error("unable to write packet", err, "addr", uf.conn.RemoteAddr().String())
	}

	// Increment internal stats
	uf.packetOutCount++
}

func (uf *UDPForwarder) appendLine(line []byte) error {
	// Make sure there is room to add the line
	if (len(uf.packet) + len(line) + 1) > uf.maxPacketSize { // Account for a newline char
		// The packet is too big... alert the caller
		return errPacketTooLarge
	}
	// If this isn't the first line in the packet, first add a newline character
	if len(uf.packet) != 0 {
		uf.packet = append(uf.packet, '\n') // Throw on a newline character
	}
	uf.packet = append(uf.packet, line...) // ... and then the line
	return nil
}

func (uf *UDPForwarder) truncatePacket() {
	uf.packet = uf.packet[:0]
}

// NoopForwarder just pulls metrics and does nothing
type NoopForwarder struct {
	queue chan *BufferLine
}

// NewNoopForwarder creates a new noop forwarder that does nothing
func NewNoopForwarder() *NoopForwarder {
	return &NoopForwarder{
		queue: make(chan *BufferLine, 50000), // arbitrarily large for now to avoid a locking slowdown
	}
}

// Run loops over the upstream Validator's queue and takes action on data
func (nf *NoopForwarder) Run() {
	for {
		// No-op
		nf.processLine(nf.getLine())
	}
}

func (nf *NoopForwarder) Stop() {}

func (nf *NoopForwarder) getLine() *BufferLine {
	return <-nf.queue
}

func (nf *NoopForwarder) processLine(bl *BufferLine) {
	bl.Done()
}

// SendLine submits a line to the NoopForwarder
func (nf *NoopForwarder) SendLine(bufferLine *BufferLine) {
	nf.queue <- bufferLine
}
