package proxy

import (
	"sync"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestAppendLine(t *testing.T) {
	fakePacket := make([]byte, 1337)[:0]
	line1 := []byte("this.is.my.metric:1|c")
	line2 := []byte("another_cool.stat:456|ms")

	uf, err := NewUDPForwarder("localhost:8125", 1337, NoopObserver{})
	require.NoError(t, err)

	assert.Equal(t, fakePacket, uf.packet)
	assert.NoError(t, uf.appendLine(line1))
	assert.Equal(t, line1, uf.packet)
	assert.NoError(t, uf.appendLine(line2))
	assert.Equal(t, append(append(line1, byte('\n')), line2...), uf.packet)
}

func TestTruncatePacket(t *testing.T) {
	uf, err := NewUDPForwarder("localhost:8125", 1337, NoopObserver{})
	require.NoError(t, err)

	assert.NoError(t, uf.appendLine([]byte("metric.metric.metric:1|c")))
	assert.NoError(t, uf.appendLine([]byte("metric.metric.metric:1|c")))
	assert.NoError(t, uf.appendLine([]byte("metric.metric.metric:1|c")))
	assert.NoError(t, uf.appendLine([]byte("metric.metric.metric:1|c")))
	assert.NoError(t, uf.appendLine([]byte("metric.metric.metric:1|c")))

	uf.truncatePacket()
	assert.Equal(t, []byte(""), uf.packet)
}

func TestSendLine(t *testing.T) {
	uf, err := NewUDPForwarder("localhost:8125", 95, NoopObserver{})
	require.NoError(t, err)

	// Create a bunch of BufferLines to process
	wg := &sync.WaitGroup{}
	for i := 0; i < 10; i++ {
		wg.Add(1)
		bl := &BufferLine{Line: []byte("10.chs:1|c"), wg: wg}
		uf.SendLine(bl)
	}

	assert.Equal(t, 10, len(uf.queue))
}

func TestProcessLine(t *testing.T) {
	uf, err := NewUDPForwarder("localhost:8125", 95, NoopObserver{})
	require.NoError(t, err)

	// Create a bunch of BufferLines to process
	wg := &sync.WaitGroup{}
	var bufferLines []*BufferLine
	for i := 0; i < 10; i++ {
		wg.Add(1)
		bl := &BufferLine{Line: []byte("10.ch:1|c"), wg: wg}
		bufferLines = append(bufferLines, bl)
	}
	// Let's process 9 of the 10 lines, which should make an 89 character packet
	// (8 lines w/ 9 characters + newline, one line w/ 9 character)
	for i := 0; i < 9; i++ {
		uf.processLine(bufferLines[i])
	}
	assert.Equal(t, 89, len(uf.packet))
	// Toss in the last packet. This should trigger a truncation, and new packet
	// that does not have a newline character at the end
	uf.processLine(bufferLines[9])
	assert.Equal(t, 9, len(uf.packet))

	// processLine should call Done() on each BufferLine when it is done w/ it
	// to test this, try calling Done() again on the WaitGroup. It should panic.
	failure := true
	defer func() {
		if r := recover(); r != nil {
			failure = false
		}
	}()
	wg.Done()
	assert.False(t, failure, "BufferLine WaitGroup did not get decremented to 0!")
}
