package proxy

import (
	"testing"
	"time"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestValidStatsd(t *testing.T) {
	v := StatsdValidator{}
	for _, c := range []struct {
		Metric string
		Valid  error
	}{
		{
			Metric: "test.test.test.test:11|c|@0.1",
			Valid:  nil,
		},
		{
			Metric: `trace.data.v1.code-justin-tv-web-web.xxx.xxx.api-platform#authenticate {:platform=>"roku"}.duration:1|c`,
			Valid:  ErrBadMetric,
		},
		{
			Metric: "Test.tEst.teSt.tesT:1.1|c",
			Valid:  nil,
		},
		{
			Metric: "test.test.test.test",
			Valid:  ErrTooFewFields,
		},
		{
			Metric: "test.test.test.test:|c",
			Valid:  ErrBadValue,
		},
		{
			Metric: "test.test.test.test:1|",
			Valid:  ErrBadType,
		},
		{
			Metric: "test.test.test.test:1||",
			Valid:  ErrIncorrectSampling,
		},
		{
			Metric: "test.test.test.test:1|c|",
			Valid:  ErrIncorrectSampling,
		},
		{
			Metric: "test.test.test.test:1|c",
			Valid:  nil,
		},
		{
			Metric: "test.test.test.test:1|m",
			Valid:  ErrBadType,
		},
		{
			Metric: "test.test.test.test:1|ms",
			Valid:  nil,
		},
		{
			Metric: "test.test.test.test:1|h",
			Valid:  nil,
		},
		{
			Metric: "test.test.test.test:1|g",
			Valid:  nil,
		},
		{
			Metric: "test.test.test.test:1|ms",
			Valid:  nil,
		},
		{
			Metric: "test.test.test.test:1|ms|",
			Valid:  ErrIncorrectSampling,
		},
		{
			Metric: "test.test.test.test:1|g|",
			Valid:  ErrIncorrectSampling,
		},
		{
			Metric: "test.test.test.test:1222|ms|@0.1",
			Valid:  nil,
		},
		{
			Metric: "chat.moving.so.fast.no.one.will.see.this.test.wont.pass.kappa.just.kidding.i.sure.do.hope.this.passes.gee.wizz.what.a.long.metric.name.please.forgive.me.statsd.i.swear.I.dont.hate.you.just.give.me.a.chance.to.pass.this.test.dont.fail.me.now.4head.elegiggle.seemsgood:1|ms",
			Valid:  nil,
		},
		{
			Metric: "test.test.test.test:1|c|@asb",
			Valid:  ErrIncorrectSampling,
		},
	} {
		_, valid := v.Validate([]byte(c.Metric))
		if valid != c.Valid {
			t.Fatalf("valid(%s) != %s instead got %s\n", c.Metric, c.Valid, valid)
		}
	}
}

func TestHashing(t *testing.T) {
	var forwarders []Forwarder
	for i := 0; i < 10; i++ {
		forwarders = append(forwarders, NewNoopForwarder())
	}

	v, err := NewStatsdValidator(nil, forwarders, "sha1", NoopObserver{}, nil)
	require.NoError(t, err)

	for _, c := range []struct {
		Metric    string
		Forwarder Forwarder
		Index     int
	}{
		{
			Metric:    "test.test.test.test",
			Forwarder: v.forwarders[8],
			Index:     8,
		},
		{
			Metric:    "foo.bar",
			Forwarder: v.forwarders[3],
			Index:     3,
		},
		{
			Metric:    "beez.baz.boz",
			Forwarder: v.forwarders[5],
			Index:     5,
		},
		{
			Metric:    "my.super-awesome.counter./.success",
			Forwarder: v.forwarders[1],
			Index:     1,
		},
		{
			Metric:    "hey.ho.lets.go",
			Forwarder: v.forwarders[3],
			Index:     3,
		},
		{
			Metric:    "many.tests.such.pass.wow",
			Forwarder: v.forwarders[3],
			Index:     3,
		},
		{
			Metric:    "chat.moving.so.fast.no.one.will.see.this.test.wont.pass.kappa.just.kidding.i.sure.do.hope.this.passes.gee.wizz.what.a.long.metric.name.please.forgive.me.statsd.i.swear.I.dont.hate.you.just.give.me.a.chance.to.pass.this.test.dont.fail.me.now.4head.elegiggle.seemsgood",
			Forwarder: v.forwarders[2],
			Index:     2,
		},
		{
			Metric:    "OHMYGODALLCAPSMETRICSARETHEBEST",
			Forwarder: v.forwarders[5],
			Index:     5,
		},
	} {
		f := v.Hash([]byte(c.Metric))
		if f != c.Forwarder {
			t.Fatalf("HashPosition(%s) != %d\n", c.Metric, c.Index)
		}
	}
}

type dummyCollector struct {
	DataPool   chan []byte
	BufferPool chan []byte
}

func (dc *dummyCollector) Run() {}
func (dc *dummyCollector) GetBuffer() []byte {
	return <-dc.DataPool
}
func (dc *dummyCollector) ReturnBuffer(buf []byte) {
	dc.BufferPool <- buf
}

func TestProcessBuffer(t *testing.T) {
	// Create a collector with some packets ready to read
	packet1 := []byte("valid.metric1:1|c\ninvalid.metric1:4:4|mmm\nvalid.metric1:4|ms|@0.1")
	packet2 := []byte("just.one.metric:3|ms")
	packet3 := []byte("only_bad_metrics:1|@2.0\n*&^%&*^%\nim.bad.at.statsd:1|r")
	// packets 4->9 all have 5 valid lines; 30 lines total
	packet4 := []byte("foo.bar:1|c\nfoo.bar:1|c\nfoo.bar:1|c\nfoo.bar:1|c\nfoo.bar:1|c")
	packet5 := []byte("foo.bar:1|c\nfoo.bar:1|c\nfoo.bar:1|c\nfoo.bar:1|c\nfoo.bar:1|c")
	packet6 := []byte("foo.bar:1|c\nfoo.bar:1|c\nfoo.bar:1|c\nfoo.bar:1|c\nfoo.bar:1|c")
	packet7 := []byte("bar.garply:1|c\nbar.garply:1|c\nbar.garply:1|c\nbar.garply:1|c\nbar.garply:1|c")
	packet8 := []byte("mostly.good.metrics:12|ms|@0.1\nmostly.good.metrics:12|ms|@0.1\none.bad.metric###:->4$c\nmostly.good.metrics:12|ms|@0.1\nmostly.good.metrics:12|ms|@0.1\nmostly.good.metrics:12|ms|@0.1\n")
	packet9 := []byte("mostly.good.metrics:12|ms|@0.1\nmostly.good.metrics:12|ms|@0.1\none.bad.metric###:->4$c\nmostly.good.metrics:12|ms|@0.1\nmostly.good.metrics:12|ms|@0.1\nmostly.good.metrics:12|ms|@0.1\n")
	dc := &dummyCollector{
		DataPool:   make(chan []byte, 10000),
		BufferPool: make(chan []byte, 10000),
	}
	dc.DataPool <- packet1
	dc.DataPool <- packet2
	dc.DataPool <- packet3

	// Create a forwarder to emit into
	nf := NewNoopForwarder()

	// Create the validator
	sv, err := NewStatsdValidator(dc, []Forwarder{nf}, "sha1", NoopObserver{}, nil)
	require.NoError(t, err)

	// Check values before starting stuff
	assert.Equal(t, 0, len(dc.BufferPool))
	assert.Equal(t, 3, len(dc.DataPool))

	// Grab a buffer and process it
	sv.processBuffer()

	// At this point there should be one less thing on the datapool
	assert.Equal(t, 0, len(dc.BufferPool))
	assert.Equal(t, 2, len(dc.DataPool))
	// Process 1 of 2 valid lines
	nf.processLine(nf.getLine())
	// Buffer still not returned to the BufferPool
	assert.Equal(t, 0, len(dc.BufferPool))
	assert.Equal(t, 2, len(dc.DataPool))

	// Process the 2nd line
	nf.processLine(nf.getLine())

	// Be sure to give the go routine time to return the buffer
	time.Sleep(1 * time.Second)

	// Now the buffer should be back in the pool
	assert.Equal(t, 1, len(dc.BufferPool))
	assert.Equal(t, 2, len(dc.DataPool))

	// Process the next packet
	sv.processBuffer()
	// Check that one less buffer is in the pool
	assert.Equal(t, 1, len(dc.BufferPool))
	assert.Equal(t, 1, len(dc.DataPool))
	// Process the line
	nf.processLine(nf.getLine())
	time.Sleep(1 * time.Second)
	// There should now be a 2nd buffer in BufferPool
	assert.Equal(t, 2, len(dc.BufferPool))
	assert.Equal(t, 1, len(dc.DataPool))

	// Process a packet with only bad lines. It should push the buffer back
	// to the BufferPool without a downstream forwarder doing anything
	sv.processBuffer()
	time.Sleep(1 * time.Second)
	assert.Equal(t, 3, len(dc.BufferPool))
	assert.Equal(t, 0, len(dc.DataPool))

	// Do a multi-packet processing round
	dc.DataPool <- packet4
	dc.DataPool <- packet5
	dc.DataPool <- packet6
	dc.DataPool <- packet7
	dc.DataPool <- packet8
	dc.DataPool <- packet9

	// 6 new buffers in the data pool
	assert.Equal(t, 3, len(dc.BufferPool))
	assert.Equal(t, 6, len(dc.DataPool))

	// Process all 6 buffers
	for i := 0; i < 6; i++ {
		sv.processBuffer()
	}
	// BufferPool should still be 3, but datapool should be empty
	assert.Equal(t, 3, len(dc.BufferPool))
	assert.Equal(t, 0, len(dc.DataPool))

	for i := 0; i < 6; i++ {
		for j := 0; j < 5; j++ {
			nf.processLine(nf.getLine())
		}
		time.Sleep(500 * time.Millisecond)
		assert.Equal(t, 3+i+1, len(dc.BufferPool))
		assert.Equal(t, 0, len(dc.DataPool))
	}
}

// Benchmark
type BenchCollector struct {
	Pool chan []byte
}

func (b BenchCollector) Run() {}
func (b BenchCollector) ReturnBuffer(buf []byte) {
	b.Pool <- buf
}
func (b BenchCollector) GetBuffer() []byte {
	return <-b.Pool
}

var testCorpus = [][]byte{
	[]byte("test.test.test.test:11|c|@0.1"),
	[]byte(`trace.data.v1.code-justin-tv-web-web.xxx.xxx.api-platform#authenticate {:platform=>"roku"}.duration:1|c`),
	[]byte("test.test.test.test:1.1|c"),
	[]byte("test.test.test.test"),
	[]byte("test.test.test.test:|c"),
	[]byte("test.test.test.test:1|"),
	[]byte("test.test.test.test:1||"),
	[]byte("test.test.test.test:1|c|"),
	[]byte("test.test.test.test:1|c"),
	[]byte("test.test.test.test:1|m"),
	[]byte("test.test.test.test:1|h"),
	[]byte("test.test.test.test:1|g"),
	[]byte("test.test.test.test:1|ms"),
	[]byte("test.test.test.test:1|ms|"),
	[]byte("test.test.test.test:1|g|"),
	[]byte("test.test.test.test:1222|ms|@0.1"),
	[]byte("test.test.test.test:1314312312|c|@.1"),
	[]byte("test.test.test.test:1|c|@asb"),
}

func BenchmarkValidation50000(b *testing.B) {
	forwarder := NewNoopForwarder()
	go forwarder.Run()
	collector := &BenchCollector{
		Pool: make(chan []byte, 1420),
	}
	go func() {
		i := 0
		for {
			collector.ReturnBuffer(testCorpus[i])
			i = (i + 1) % len(testCorpus)
		}
	}()

	validator, err := NewStatsdValidator(collector, []Forwarder{forwarder}, "sha1", NoopObserver{}, nil)
	require.NoError(b, err)

	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		validator.processBuffer()
	}
}
