# Obtains the name of the AWS region configured by the inherited AWS provider.
data "aws_region" "current" {}

# This grabs the latest recommended ECS-optimized Amazon Linux 2 image,
# so that we can use the image in the autoscaling group's launch template.
#
# This means whenever the autoscaling group launches a new instance,
# it will run the latest recommended image available.
#
# This helps keep the system up-to-date in a semi-automated fashion.
#
# https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html
data "aws_ssm_parameter" "ecs_optimized_ami_id" {
  name = "/aws/service/ecs/optimized-ami/amazon-linux-2/recommended/image_id"
}

# The ECS container agent makes calls to the ECS API on our behalf.
#
# Container instances that run the agent require an IAM policy and role for the service to know
# that the agent belongs to us.
#
# Before we can launch container instances and register them into a cluster, we must create an
# IAM role for those container instances to use when they are launched.
#
# Note: this requirement applies to container instances launched with the ECS-optimized AMI
# provided by Amazon, and only with the EC2 launch type.
#
# https://docs.aws.amazon.com/AmazonECS/latest/developerguide/instance_IAM_role.html
resource "aws_iam_role" "container_instance" {
  name_prefix        = "instance_${var.name}"
  assume_role_policy = data.aws_iam_policy_document.container_instance.json
}

data "aws_iam_policy_document" "container_instance" {
  statement {
    principals {
      type        = "Service"
      identifiers = ["ec2.amazonaws.com"]
    }

    actions = ["sts:AssumeRole"]
  }
}

resource "aws_iam_instance_profile" "container_instance" {
  name_prefix = "instance_${var.name}"
  role        = aws_iam_role.container_instance.name
}

# Allow EC2 instance to function as an ECS container instance.
resource "aws_iam_role_policy_attachment" "container_instance_ecs" {
  role       = aws_iam_role.container_instance.name
  policy_arn = "arn:aws:iam::aws:policy/service-role/AmazonEC2ContainerServiceforEC2Role"
}

# Allow container instance to be managed by SSM.
resource "aws_iam_role_policy_attachment" "container_instance_ssm" {
  role       = aws_iam_role.container_instance.name
  policy_arn = "arn:aws:iam::aws:policy/AmazonSSMManagedInstanceCore"
}

# The launch template instructs the autoscaling group which kind
# of instances should be created, when necessary.
resource "aws_launch_template" "proxy" {
  name_prefix            = var.name
  description            = "Launch template for ${var.name} ECS container instances."
  image_id               = data.aws_ssm_parameter.ecs_optimized_ami_id.value
  instance_type          = var.asg_instances.type
  vpc_security_group_ids = var.vpc_security_group_ids
  ebs_optimized          = true
  user_data              = base64encode(templatefile("${path.module}/userdata.sh", { cluster_name = var.name }))

  iam_instance_profile {
    arn = aws_iam_instance_profile.container_instance.arn
  }

  monitoring {
    enabled = true
  }
}

# This is the autoscaling group which backs the cluster capacity provider.
# The capacity provider will use this group to provide instance capacity
# to the ECS cluster.
resource "aws_autoscaling_group" "proxy" {
  name_prefix = var.name
  max_size    = var.asg_instances.max
  min_size    = 0 # Set 0 to allow the ECS Capacity Provider full autonomy over the ASG.

  vpc_zone_identifier       = var.subnet_ids
  health_check_grace_period = 30 # seconds
  default_cooldown          = 30 # seconds
  protect_from_scale_in     = true

  launch_template {
    id      = aws_launch_template.proxy.id
    version = aws_launch_template.proxy.latest_version
  }

  tag {
    key   = "Name"
    value = var.name

    propagate_at_launch = true
  }

  tag {
    key   = "Environment"
    value = var.environment

    propagate_at_launch = true
  }
}

# This cluster capacity provider enables the ECS cluster to add container instances as needed.
resource "aws_ecs_capacity_provider" "proxy" {
  name = "provider_${var.name}"

  auto_scaling_group_provider {
    auto_scaling_group_arn         = aws_autoscaling_group.proxy.arn
    managed_termination_protection = "ENABLED"

    managed_scaling {
      status          = "ENABLED"
      target_capacity = 100
    }
  }
}

# The ECS cluster is the logical grouping of services.
resource "aws_ecs_cluster" "proxy" {
  name               = var.name
  capacity_providers = [aws_ecs_capacity_provider.proxy.name]
  default_capacity_provider_strategy {
    capacity_provider = aws_ecs_capacity_provider.proxy.name
  }

  setting {
    name  = "containerInsights"
    value = "enabled"
  }
}

# This log group provides a destination for the ECS logs to land.
resource "aws_cloudwatch_log_group" "proxy" {
  name              = var.name
  retention_in_days = var.log_retention_days
}

# The repository that stores the application images in ECR.
resource "aws_ecr_repository" "proxy" {
  name                 = var.name
  image_tag_mutability = "IMMUTABLE"
}

# This lifecycle policy describes how many images to keep in the repository.
resource "aws_ecr_lifecycle_policy" "proxy" {
  repository = aws_ecr_repository.proxy.name
  policy     = <<EOF
{
  "rules": [
    {
      "rulePriority": 1,
      "description": "Keep last ${var.max_images} images",
      "selection": {
        "tagStatus": "any",
        "countType": "imageCountMoreThan",
        "countNumber": ${var.max_images}
      },
      "action": {
        "type": "expire"
      }
    }
  ]
}
EOF
}

resource "aws_iam_role" "task" {
  name_prefix        = "task_${var.name}"
  assume_role_policy = data.aws_iam_policy_document.task.json
}

data "aws_iam_policy_document" "task" {
  statement {
    principals {
      type        = "Service"
      identifiers = ["ecs-tasks.amazonaws.com"]
    }

    actions = ["sts:AssumeRole"]
  }
}

# Allow ECS tasks to put metric data to CloudWatch.
data "aws_iam_policy_document" "put_cloudwatch_metrics" {
  statement {
    actions   = ["cloudwatch:PutMetricData"]
    resources = ["*"]
  }
}

resource "aws_iam_role_policy" "put_cloudwatch_metrics" {
  name   = "put_cloudwatch_metrics"
  role   = aws_iam_role.task.name
  policy = data.aws_iam_policy_document.put_cloudwatch_metrics.json
}


# Allow ECS tasks to fetch configuration about our stats hosts
data "aws_iam_policy_document" "fetch_host_config" {
  statement {
    actions   = ["ec2:DescribeInstances"]
    resources = ["*"]
  }

  statement {
    actions   = ["ecs:ListContainerInstances"]
    resources = [var.stats_cluster_arn]
  }

  # ListTasks can only be limited with a Condition, not a resource :/
  # https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticcontainerservice.html
  statement {
    actions   = ["ecs:DescribeContainerInstances", "ecs:ListTasks", "ecs:DescribeTasks"]
    resources = ["*"]
    condition {
      variable = "ecs:cluster"
      test     = "ArnLike"
      values   = [var.stats_cluster_arn]
    }
  }
}

resource "aws_iam_role_policy" "fetch_host_config" {
  name   = "fetch_host_config"
  role   = aws_iam_role.task.name
  policy = data.aws_iam_policy_document.fetch_host_config.json
}

# The Amazon ECS container agent makes calls to the Amazon ECS API
# on our behalf.
#
# The agent requires an IAM role for the service to know that the agent
# belongs to us.
#
# This IAM role is referred to as a task execution role.
#
# https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_execution_IAM_role.html
resource "aws_iam_role" "execution" {
  name_prefix        = "execution_${var.name}"
  assume_role_policy = data.aws_iam_policy_document.execution.json
}

data "aws_iam_policy_document" "execution" {
  statement {
    actions = ["sts:AssumeRole"]

    principals {
      type        = "Service"
      identifiers = ["ecs-tasks.amazonaws.com"]
    }
  }
}

resource "aws_iam_role_policy_attachment" "execution" {
  role       = aws_iam_role.execution.name
  policy_arn = "arn:aws:iam::aws:policy/service-role/AmazonECSTaskExecutionRolePolicy"
}


resource "aws_iam_role" "service" {
  name_prefix        = "service_${var.name}"
  assume_role_policy = data.aws_iam_policy_document.service.json
}

data "aws_iam_policy_document" "service" {
  statement {
    principals {
      type        = "Service"
      identifiers = ["ecs.amazonaws.com"]
    }

    actions = ["sts:AssumeRole"]
  }
}

# Allow ECS services to register/deregister with a load balancer.
resource "aws_iam_role_policy_attachment" "service" {
  role       = aws_iam_role.service.name
  policy_arn = "arn:aws:iam::aws:policy/service-role/AmazonEC2ContainerServiceRole"
}



# The task definition provides a blueprint for how the application task should be run.
resource "aws_ecs_task_definition" "proxy" {
  family             = var.name
  task_role_arn      = aws_iam_role.task.arn
  execution_role_arn = aws_iam_role.execution.arn
  network_mode       = "host"

  container_definitions = <<EOF
[{
  "name": "${var.name}",
  "essential": true,
  "image": "${aws_ecr_repository.proxy.repository_url}:${var.image_tag}",
  "cpu": ${var.reservation.cpu_units},
  "memoryReservation": ${var.reservation.memory_mib},
  "portMappings": [{
    "protocol": "udp",
    "containerPort": ${var.udp_port},
    "hostPort": ${var.udp_port}
  }, {
    "protocol": "tcp",
    "containerPort": ${var.tcp_port},
    "hostPort": ${var.tcp_port}
  }],
  "logConfiguration": {
    "logDriver": "awslogs",
    "options": {
      "awslogs-region": "${data.aws_region.current.name}",
      "awslogs-group": "${var.name}",
      "awslogs-stream-prefix": "${var.name}"
    }
  },
  "environment": [
    {"name": "ADDRESSES", "value": "${join(",", var.addresses)}"},
    {"name": "AWS_REGION", "value": "${data.aws_region.current.name}"},
    {"name": "ENVIRONMENT", "value": "${var.environment}"},
    {"name": "HASH", "value": "${var.hash_algorithm}"},
    {"name": "HEALTH_PORT", "value": "${var.tcp_port}"},
    {"name": "STATS_PORT", "value": "${var.udp_port}"},
    {"name": "NUM_WORKERS", "value": "${var.num_forwarders}"},
    {"name": "UPSTREAM_CLUSTER", "value": "${var.stats_cluster_name}"},
    {"name": "UPSTREAM_SERVICE", "value": "${var.stats_service_name}"}
  ],
  "ulimits": [{
    "name": "nofile",
    "softLimit": ${var.ulimit},
    "hardLimit": ${var.ulimit}
  }]
}]
EOF
}

resource "aws_ecs_service" "proxy" {
  name            = var.name
  cluster         = aws_ecs_cluster.proxy.arn
  task_definition = aws_ecs_task_definition.proxy.arn

  desired_count = var.asg_instances.desired
  iam_role      = aws_iam_role.service.arn
  depends_on    = [aws_iam_role.service]

  enable_ecs_managed_tags = true

  capacity_provider_strategy {
    capacity_provider = aws_ecs_capacity_provider.proxy.name

    weight = 1
  }

  load_balancer {
    target_group_arn = aws_lb_target_group.proxy.arn
    container_name   = var.name
    container_port   = var.udp_port
  }

  ordered_placement_strategy {
    type  = "spread"
    field = "instanceId"
  }

  deployment_maximum_percent         = 200
  deployment_minimum_healthy_percent = 100
}

resource "aws_lb" "proxy" {
  load_balancer_type = "network"

  name     = var.name
  subnets  = var.subnet_ids
  internal = true

  enable_deletion_protection = true
}

resource "aws_lb_target_group" "proxy" {
  name                 = var.name
  target_type          = "instance"
  protocol             = "UDP"
  port                 = var.udp_port
  vpc_id               = var.vpc_id
  deregistration_delay = 60

  health_check {
    protocol = "HTTP"
    port     = var.tcp_port
    path     = "/health"
    interval = 10 # seconds
  }
}

resource "aws_lb_listener" "proxy" {
  load_balancer_arn = aws_lb.proxy.arn
  protocol          = "UDP"
  port              = var.udp_port

  default_action {
    type             = "forward"
    target_group_arn = aws_lb_target_group.proxy.arn
  }
}

resource "aws_cloudwatch_metric_alarm" "invalid_lines" {
  alarm_name          = "go_statsd_proxy_invalid_lines"
  alarm_description   = "Alarm when go statsd proxy's invalid line count is nonzero. https://wiki.xarth.tv/display/VID/SHG+Metrics+and+Alarming#SHGMetricsandAlarming-go_statsd_proxy_invalid_lines"
  comparison_operator = "GreaterThanOrEqualToThreshold"
  metric_name         = "InvalidLines"
  statistic           = "Sum"
  namespace           = "/go-statsd-proxy"
  dimensions = {
    Region = data.aws_region.current.name
  }

  evaluation_periods = "5"
  period             = "60"
  threshold          = "0"
  alarm_actions      = var.pagerduty_sns_topic_arns
  ok_actions         = var.pagerduty_sns_topic_arns
  treat_missing_data = "missing"
}


resource "aws_cloudwatch_metric_alarm" "cpu_utilization" {
  alarm_name          = "go_statsd_proxy_cpu_utilization"
  alarm_description   = "Alarm when go statsd proxy's cpu utilization is > 80% for 5 minute. https://wiki.xarth.tv/display/VID/SHG+Metrics+and+Alarming#SHGMetricsandAlarming-go_statsd_proxy_cpu_utilization"
  comparison_operator = "GreaterThanOrEqualToThreshold"
  metric_name         = "CPUUtilization"
  statistic           = "Average"
  namespace           = "AWS/ECS"
  dimensions = {
    ClusterName = var.name
    ServiceName = var.name
  }

  evaluation_periods = "5"
  period             = "60"
  threshold          = "80"
  alarm_actions      = var.pagerduty_sns_topic_arns
  ok_actions         = var.pagerduty_sns_topic_arns
  treat_missing_data = "missing"
}

resource "aws_cloudwatch_metric_alarm" "memory_utilization" {
  alarm_name          = "go_statsd_proxy_memory_utilization"
  alarm_description   = "Alarm when go statsd proxy's memory utilization is > 80% for 5 minute. https://wiki.xarth.tv/display/VID/SHG+Metrics+and+Alarming#SHGMetricsandAlarming-go_statsd_proxy_memory_utilization"
  comparison_operator = "GreaterThanOrEqualToThreshold"
  metric_name         = "MemoryUtilization"
  statistic           = "Average"
  namespace           = "AWS/ECS"
  dimensions = {
    ClusterName = var.name
    ServiceName = var.name
  }

  evaluation_periods = "5"
  period             = "60"
  threshold          = "80"
  alarm_actions      = var.pagerduty_sns_topic_arns
  ok_actions         = var.pagerduty_sns_topic_arns
  treat_missing_data = "missing"
}
