#!/usr/bin/env python

import sys
import datetime
import uuid
import botocore
import boto3

DIMENSIONS = [{"Name": "ip", "Value": sys.argv[2]}]
MAX_METRICS_DEFAULT = 128
MIN_METRICS = 8
AWS_DEFAULT_REGION = "us-west-2"
NAMESPACE = "Infoblox"
AWS_ROLE_NAME = "arn:aws:iam::277437172290:role/CloudwatchAgentMetrics"

class CloudwatchHelper():
    def _refresh(self):
        " Refresh tokens by calling assume_role again "
        boto3.DEFAULT_SESSION = None
        params = {
            "RoleArn": AWS_ROLE_NAME,
            "RoleSessionName": str(uuid.uuid4()),
            "DurationSeconds": 900,
        }
        sts_client = boto3.client("sts", region_name=self.aws_region)
        response = sts_client.assume_role(**params).get("Credentials")
        credentials = {
            "access_key": response.get("AccessKeyId"),
            "secret_key": response.get("SecretAccessKey"),
            "token": response.get("SessionToken"),
            "expiry_time": response.get("Expiration").isoformat(),
        }
        return credentials

    def create_cw_connection(self):
        session_credentials = botocore.credentials.RefreshableCredentials.create_from_metadata(
            metadata=self._refresh(),
            refresh_using=self._refresh,
            method="sts-assume-role",
        )
        session = botocore.session.get_session()
        session._credentials = session_credentials
        session.set_config_variable("region", self.aws_region)
        autorefresh_session = boto3.Session(botocore_session=session)
        self.cw_connection = autorefresh_session.client('cloudwatch')

    def publish_metrics(self):
        while True:
            try:
                self.cw_connection.put_metric_data(
                    Namespace=NAMESPACE,
                    MetricData=self.metrics[:self.max_metrics]
                )
                print("Published {} metrics".format(self.max_metrics))
                self.metrics = self.metrics[self.max_metrics:]
                self.max_metrics += 1
                return
            except botocore.exceptions.ClientError as ce:
                if ce.response['Error']['Code'] == 'RequestEntityTooLarge':
                    if self.max_metrics <= MIN_METRICS:
                        break
                    self.max_metrics -= MIN_METRICS
        self.metrics = self.metrics[self.max_metrics:]

    def run(self):
        self.aws_region = AWS_DEFAULT_REGION
        self.metrics = []
        self.max_metrics = MAX_METRICS_DEFAULT
        self.create_cw_connection()
        with (open(sys.argv[1])) as f:
            for line in f.readlines():
                name, value = line.strip().split()
                self.metrics.append({
                        'MetricName': name,
                        'Dimensions': DIMENSIONS,
                        'Timestamp': datetime.datetime.utcnow(),
                        'Value': float(value)
                    })
                while len(self.metrics) >= self.max_metrics:
                    self.publish_metrics()
        while self.metrics:
            self.publish_metrics()

CloudwatchHelper().run()
