#!/usr/bin/env python
import getpass
import os
import socket
import subprocess
import sys
import time


def set_pxe_boot(hostname):
    run_ipmi_command(
        hostname,
        'ipmi-chassis-config',
        '--commit -e Chassis_Boot_Flags:Boot_Device=PXE',
    )


def is_ip_addr(ip_addr):
    try:
        socket.inet_aton(ip_addr)
        return True
    except socket.error:
        return False


def get_ipmi_addr(hostname):
    if is_ip_addr(hostname) or 'ipmi' in hostname:
        return hostname
    else:
        sys.stderr.write("{0} is not a valid IPMI address.\n".format(hostname))
        sys.stderr.write('Valid addresses are IPs or hostnames containing \'-ipmi\'.\n')
        raise RuntimeError("{0} is not a valid IPMI address.".format(hostname))

def get_ipmi_password():
    ipmi_password = os.environ.get('IPMI_PASSWORD', None)
    if ipmi_password:
        return ipmi_password
    try:
        return subprocess.check_output(["pass", "show", "twitch/systems/ipmi"]).strip()
    except:
        return getpass.getpass()


def run_ipmi_command(hostname, command, flags, get_output=False):
    ipmi_password = get_ipmi_password()
    command = '/usr/bin/env {0} -h {1} -u ADMIN -p {2} {3}'.format(
        command,
        get_ipmi_addr(hostname),
        ipmi_password,
        flags,
    )
    out = subprocess.check_output(
        command.split(),
        stderr=subprocess.STDOUT,  # 2>&1
    )
    if get_output:
        return out
    if out:
        sys.stdout.write(out)


def run_ipmi_console(hostname):
    ipmi_password = get_ipmi_password()
    command = [
        "ipmiconsole",
        "-h",
        get_ipmi_addr(hostname),
        "-u",
        "ADMIN",
        "-p",
        ipmi_password,
    ]
    os.execvp("ipmiconsole", command)


def ipmi_console(hostname, deactivate=False):
    if deactivate:
        run_ipmi_command(hostname, "ipmiconsole", "--deactivate")
        time.sleep(2)
        run_ipmi_console(hostname)
    else:
        run_ipmi_console(hostname)


def ipmi_uid(hostname, time):
    if time < 0:
        run_ipmi_command(hostname, "ipmi-chassis", "--chassis-identify=FORCE")
    elif time == 0:
        # setting it to 0 doesn't always turn it off. By setting it to 1 it
        # will be on for one second and then turn off. The minimum amount of
        # time.
        run_ipmi_command(hostname, "ipmi-chassis", "--chassis-identify=%d" % 1)
    else:
        run_ipmi_command(
            hostname,
            "ipmi-chassis",
            "--chassis-identify=%d" % time
        )


def ipmi_datamac(hostname):
    data = run_ipmi_command(hostname, "ipmi-raw", "0 30 21", get_output=True)
    split_data = data.rstrip().split(" ")
    if len(split_data) != 13:
        m = "Can't parse mac address. Unknown information: '{0}'".format(data)
        print >> sys.stderr, m
        exit(-1)
    else:
        return ":".join(split_data[7:])


def ipmi_fru(hostname):
    run_ipmi_command(
        hostname,
        "ipmi-fru",
        "--skip-checks --quiet-cache --ignore-sdr-cache"
    )


def ipmi_power(hostname, command):
    run_ipmi_command(hostname, "ipmipower", command)


def ipmi_lan(hostname):
    run_ipmi_command(hostname, "bmc-config", "--checkout --section=Lan_Conf")


