package tv.justin.vidcs;

// helper function for common head -n1
def readLine(s) {
	return readFile(s).trim().split("\n")[0]
}

// PackageFromDir reads input files (if exists) to create `fpm` command line arguments
// it will then run fpm to create a package, based on `package_type` passed in (defaults to "deb")
def PackageFromDir(package_type = "deb", source = "fpm_source", fpm_scripts = "fpm_scripts", version = "1.0.0", arch = "amd64") {

	params = ""

	/*
	Check for any files in the local repo which set custom FPM params
	*/
	dir(fpm_scripts) {
		args = makeArgMap()
		hook_args = makeHookArgs(package_type)

		// special case for filenames not matching long argument
		if (fileExists("ARCH")) {
			args["architecture"] = readFile("ARCH").trim()
		}

		if (fileExists("${package_type.toUpperCase()}_PARAMS")) {
			params = readLine("${package_type.toUpperCase()}_PARAMS")
		}

		if (fileExists("COMMON_PARAMS")) {
			params = "${params} ${readLine("COMMON_PARAMS")}"
		}

		if (fileExists("${package_type.toUpperCase()}_DEPENDENCIES")) {
			dependencies = readFile("${package_type.toUpperCase()}_DEPENDENCIES").trim().split("\n")
			dependencies_args = dependencies.collect {e -> "-d '$e'"}.join(" ")
		} else {
			dependencies_args = ""
		}
	}

	/*
	Some Validation before we create a package
	*/
	if (!args.containsKey("name")) {
		args["name"] = env.JOB_NAME
	}

	if (!args.containsKey("version")) {
		args["version"] = version
	}

	if (!args.containsKey("iteration")) {
		args["iteration"] = env.BUILD_NUMBER
	}

	if (!(args["version"] ==~ /^([0-9]+)\.([0-9]+)\.([0-9]+)$/)) {
		err = "invalid version number: ${args.version}"
		echo err
		error err
	}

	if (!(args["name"] ==~ /^[a-zA-Z0-9_-]+$/)) {
		err = "invalid name: ${args.name}"
		echo err
		error err
	}

	// turn into arguments for fpm
	fpm_args = args.collect { key, value -> "--${key} '${value}'" }.join(" ")

	// run it
	sh(
		script: "fpm -s dir -t ${package_type} -C ${source} ${fpm_args} ${params} ${hook_args} ${dependencies_args}"
	)

/*
XXX: this is here for reference, and missing pieces not implemented yet.
# check for special DEB_PRE_DEPENDS or DEB_BUILD_DEPENDS files
if [[ -f "DEB_PRE_DEPENDS" ]]; then
  DEPS="\${DEPS} --deb-pre-depends \"\$(cat DEB_PRE_DEPENDS)\""
fi
if [[ -f "DEB_BUILD_DEPENDS" ]]; then
  DEPS="\${DEPS} --deb-build-depends \"\$(cat DEB_BUILD_DEPENDS)\""
fi
MISC_PARAMS=""
# Add other misc parameters if special files exist
if [[ -f LICENSE ]]; then
  MISC_PARAMS="--license \\"\$(cat LICENSE)\\""
fi
if [[ -f RPM_SUMMARY && "$package_type" == "rpm" ]]; then
  MISC_PARAMS="\${MISC_PARAMS} --rpm-summary \\"\$(cat RPM_SUMMARY)\\""
fi
if [[ -f CHANGELOG ]]; then
  MISC_PARAMS="\${MISC_PARAMS} --$package_type-changelog \${this_dir}/CHANGELOG"
fi
if [[ -f DESCRIPTION ]]; then
  MISC_PARAMS="\${MISC_PARAMS} --description \\"\$(cat DESCRIPTION)\\""
fi
CMD="fpm -s dir -t ${package_type} -p ${package_output} -n \${NAME:-\${WORKSPACE}} -v \${VERSION:-$version} --iteration \${ITERATION} -a \${ARCH:-$arch} \${script_args} \${PKG_PARAMS} \${DEPS} \${MISC_PARAMS} \${COMMON_PARAMS} -p \${WORKSPACE} -C $source"
echo "Running command: '\${CMD}'"
eval "\${CMD}"
"""
*/
}

def makeArgMap() {
	standard_args = ["name", "version", "iteration", "description", "maintainer", "url", "license"]
	arg_map = [:]
	for (arg in standard_args) {
		if (fileExists(arg.toUpperCase())) {
			arg_map[arg] = readFile(arg.toUpperCase()).trim()
		}
	}
	return arg_map
}

// checks if a hook script exists first in a package_type specific subdirectory, then checks the base dir
// return map of fpm arguments for discovered hooks
def makeHookArgs(package_type) {
	hook_list = ["before-install", "before-remove", "before-upgrade", "after-install", "after-remove", "after-upgrade"]
	hook_map = [:]
	for (hook in hook_list) {
		// first check for any package_type specific hooks
		if (fileExists("${package_type}/${hook}")) {
			hook_map[hook] = "${pwd()}/${package_type}/${hook}"
		} else if (fileExists(hook)) {
			hook_map[hook] = "${pwd()}/${hook}"
		}
	}

	// return a string of arguments, properly quoted
	return hook_map.collect { key, value -> "--${key} '${value}'"}.join(" ")
}
