from chalice import Chalice
import boto3
import random
import json
import requests
import urllib3
import ipaddress

''' API for interaction with Infoblox to create/retrieve fixedaddress dhcp reservations
   for kontrons.
   NOTE: Interaction with Infoblox Should 0-pad the 'rackunit' (Chassis) field due to convention
         Interaction with the BuildSheet service should NOT 0-pad '''

urllib3.disable_warnings()

app = Chalice(app_name='kontron_lambda')

SSMPATHS = {
         'username': '/kontron/infoblox/username',
         'password': '/kontron/infoblox/password',
         }

bs_url = 'buildsheet-service.us-west-2.prod.provisioner.live-video.a2z.com/twirp/code.justin.tv.videotools.buildsheetservice.api.buildsheet.BuildSheetService/BuildSheet'
ib_url = 'gm-infoblox.twitch.tv'
api_path = 'wapi/v2.5'


class entry(object):
    def __init__(self, comment='created by lambda', ipv4addr=None, mac=None, enable_ddns=True, ddns_hostname=None, subnet=None, extattrs=None):
        self.comment = comment
        self.mac = mac
        self.enable_ddns = enable_ddns
        self.ipv4addr = ipv4addr or self.getipv4addr(subnet)
        self.ddns_hostname = ddns_hostname or self.getHostname()
        self.extattrs = extattrs

    def toJson(self):
        return json.dumps(self, default=lambda o: o.__dict__, sort_keys=True, indent=4)

    def setipv4addr(self, ipv4=None, subnet=None):
        if ipv4 is None:
            self.ipv4addr = self.getipv4addr(subnet)
        else:
            self.ipv4addr = ipv4

    def getipv4addr(self, subnet):
        if not subnet:
            # TODO: Is this safe?
            return '.'.join(str(random.randint(0, 255)) for _ in range(4))
        else:
            return 'func:nextavailableip:{}'.format(subnet)

    def getHostname(self):
        return 'mgmt-rack.chassis.slotID'


@app.route('/')
def index():
    return {'hello': 'world'}


@app.route('/auth', methods=['GET'], api_key_required=True)
def auth():
    return {'auth': 'OK'}


@app.route('/credscheck', methods=['GET'], api_key_required=True)
def credscheck():
    username, password = getCredentials()
    return {'username': username, 'password': password}


def getCredentials():
    ssm = boto3.client('ssm')
    username = ssm.get_parameter(Name=SSMPATHS['username'], WithDecryption=False)
    password = ssm.get_parameter(Name=SSMPATHS['password'], WithDecryption=True)
    return (username['Parameter']['Value'], password['Parameter']['Value'])


# strips all but digits, 0pads to 2 chars, returns str
def fix_rackunit(ru):
    ru = ''.join([c for c in ru if c in '0123456789'])
    ru = "{:0>2}".format(ru)
    return(ru)


# strips all but digits, prepends 'r'
def fix_rack(rack):
    rack = ''.join([c for c in rack if c in '0123456789'])
    rack = "r{}".format(rack)
    return(rack)


@app.route('/list/{pop}/{rack}/{rackunit}', methods=['GET'], api_key_required=True)
def api_list(pop, rack, rackunit):
    rack = fix_rack(rack)
    rackunit = fix_rackunit(rackunit)
    refs = get_existing_refs(pop, rack, rackunit)
    numrefs = num_refs(pop, rack, rackunit, refs)
    if numrefs > 0:
        return {'status': 'Noop', 'message': '{} records already exist for {}/{}/{}'.format(numrefs, pop, rack, rackunit), 'output': refs}
    else:
        objects = get_bs_entries(pop, rack, rackunit)
        if not isinstance(objects, type([])):
            if 'status' in objects and objects['status'] == 'Error':
                return {'status': 'Error', 'message': 'No records exist for {}/{}/{}, No buildsheet info available.'.format(pop, rack, rackunit)}
        retval = list(map(lambda o: o.__dict__, objects))
        return {'status': 'Error', 'message': 'No records exist for {}/{}/{}, outputting Buildsheet verification.'.format(pop, rack, rackunit), 'output': retval}


def get_existing_refs(pop, rack, rackunit):
    print('fetching existing records for pop: {}, rack: {}, rackunit: {}'.format(pop, rack, rackunit))
    username, password = getCredentials()
    r = requests.get('https://{}/{}/fixedaddress?*Site={}&*RackNumber={}&*Chassis={}&_return_fields=ipv4addr,mac,ddns_hostname,extattrs'.format(ib_url, api_path, pop, rack, rackunit), verify=False, auth=(username, password))
    if len(r.json()) == 0:
        if int(rackunit) < 10 and int(rackunit) > 0 and '0' in str(rackunit):
            print('Checking possible non-0padded rackunit {}'.format(rackunit))
            return get_existing_refs(pop, rack, int(rackunit))
        return {'status': 'Error', 'message': 'ERR, no records found'}
    return r.json()


@app.route('/exist/{pop}/{rack}/{rackunit}', methods=['GET'], api_key_required=True)
def exists(pop, rack, rackunit):
    rack = fix_rack(rack)
    rackunit = fix_rackunit(rackunit)
    num = num_refs(pop, rack, rackunit)
    if num > 0:
        return {'status': 'Noop', 'message': '{} records already exist for {}/{}/{}'.format(num, pop, rack, rackunit)}
    else:
        return {'status': 'Noop', 'message': 'no records found for {}/{}/{}'.format(pop, rack, rackunit)}


# Returns Number of existing refs from infoblox (or a given refs json)
def num_refs(pop, rack, rackunit, refs=None):
    if refs is None:
        refs = get_existing_refs(pop, rack, rackunit)
    if ('status' in refs and refs['status'] == 'Error') or ('Error' in refs):
        return 0
    else:
        return len(refs)


# Takes slot identifier, returns dict of node_type, bmc_name, number, dns_key
def slot_type(slot):
    slot_info = {}
    if slot[0] in 'sS':
        slot_info['dns_key'] = 's'
        slot_info['node_type'] = 'nodes'
        slot_info['bmc_name'] = 'Base_MAC_BMC'
    elif slot[0] in 'hH':
        slot_info['dns_key'] = 'h'
        slot_info['node_type'] = 'hub_nodes'
        slot_info['bmc_name'] = 'Base_MAC_ShMC'
    else:
        return {'status': 'Error', 'error': 'Slot must start with an h (hubnode) or s (node)'}
    slot_info['number'] = slot[1:]
    return slot_info


# Requests buildsheet data from buildsheet service, returns it
def get_buildsheet(pop, rack, rackunit):
    try:
        rackunit = int(rackunit)
        data = {'location': {'pop': pop, 'rack': rack, 'rack_unit': rackunit}}
        h = {'Content-type': 'application/json'}
        r = requests.post('https://{}'.format(bs_url), headers=h, data=json.dumps(data))
        buildsheet = r.json()['build_sheet']
    except Exception:
        return None
    return buildsheet


# creates entry objects from buildsheet data, validates pop/rack/ru has a network in ib
#   validates network has a dhcprange in ib
def get_bs_entries(pop, rack, rackunit):
    network = get_infoblox_network(pop, rack)
    if network is None:
        return {'status': 'Error', 'error': 'Network object not found'}
    dhcprange = get_infoblox_dhcprange(network)
    if dhcprange is None:
        return {'status': 'Error', 'error': 'DHCPrange object not found'}
    objects = []
    buildsheet = get_buildsheet(pop, rack, rackunit)
    if buildsheet is None:
        return {'status': 'Error', 'error': 'Buildsheet service unavailable or buildsheet does not exist'}
    hubnodes = get_nodes(pop, rack, rackunit, buildsheet, dhcprange, 'hub_nodes')
    objects.extend(hubnodes)
    nodes = get_nodes(pop, rack, rackunit, buildsheet, dhcprange, 'nodes')
    objects.extend(nodes)
    return objects


@app.route('/create/{pop}/{rack}/{rackunit}', methods=['GET'], api_key_required=True)
def create(pop, rack, rackunit):
    rack = fix_rack(rack)
    rackunit = fix_rackunit(rackunit)
    refs = get_existing_refs(pop, rack, rackunit)
    numrefs = num_refs(pop, rack, rackunit, refs)
    if numrefs > 0:
        return {'status': 'Noop', 'message': '{} records already exist for {}/{}/{}'.format(numrefs, pop, rack, rackunit), 'output': refs}
    print('pushing pop: {}, rack: {}, rackunit: {}'.format(pop, rack, rackunit))
    objects = get_bs_entries(pop, rack, rackunit)
    # check for errors
    if len(objects) == 0:
        return {'status': 'Error', 'error': 'No nodes found in buildsheet for {}/{}/{}'.format(pop, rack, rackunit)}
    if not isinstance(objects, type([])):
        if 'status' in objects and objects['status'] == 'Error':
            print('getting buildsheet or network info failed, forwarding status and error to caller')
            return objects
    # Make sure infoblox has a network for the rack
    network = get_infoblox_network(pop, rack)
    if network is None:
        return {'status': 'Fail', 'error': 'Network object not found'}
    available_space = get_available_space(network)
    # Check for nodes already holding leases, count so we can skip
    already = 0
    for obj in objects:
        if obj.ipv4addr is None:
            next
        elif obj.ipv4addr == 'FAIL':
            # Since we filter on active leases already, this shouldn't alert unless its really a problem
            # Try to fix again, restricting to the active network
            already += fix_preassigned(obj, network)
        else:
            already += 1
    print('have {} objects ({} already assigned) and space is {}'.format(len(objects), already, available_space))
    if (len(objects) - already) > available_space:
        return {'status': 'Fail', 'error': 'Subnet/dhcprange does not have enough available addresses'}
    refs = push_infoblox_list(objects)
    print(refs)
    return {'status': 'OK', 'refs': refs}


# fixes pre-assigned multiple leases for entry objs, returns 1 if one lease is left
def fix_preassigned(obj, subnet):
    # See if any active leases are in the current subnet and assign it
    leases = active_leases(fetch_current_leases(obj.mac))
    ips = addresses_from_leases(leases)
    for ip in (ips):
        if ipaddress.ip_address(ip) in ipaddress.ip_network(subnet):
            obj.setipv4addr(ip)
            print('IP {} is in our subnet {}, updated obj for mac {}'.format(obj.ipv4addr, subnet, obj.mac))
            return 1
    obj.setipv4addr(None)
    return 0


# Fetch leases assigned to given Mac, return list of refs
def fetch_current_leases(mac):
    username, password = getCredentials()
    r = requests.get('https://{}/{}/lease?hardware={}&_return_fields=address,binding_state'.format(ib_url, api_path, mac), verify=False, auth=(username, password))
    leases = r.json()
    if len(leases) < 1:
        # no leases are found
        print('no lease found for {}'.format(mac))
        return []
    return leases


# Filter leases to only "ACTIVE" or "STATIC" or "OFFERED" ones
# Note: this could still include several of the same MAC, Address pairs as infoblox returns a "lease object"
# per member of its grid
def active_leases(leases):
    active_ls = []
    for lease in leases:
        if lease['binding_state'] not in ['STATIC', 'ACTIVE', 'OFFERED']:
            next
        active_ls.append(lease)
    return active_ls


# takes list of lease refs, returns set of ip addrs
def addresses_from_leases(leases):
    ips = set()
    for lease in leases:
        ips.add(lease['address'])
    return ips


# Checks BuildSheet vs InfoBlox refs for a given sled, updates mac for the fixedaddr
@app.route('/replace/{pop}/{rack}/{rackunit}/{slot}', methods=['GET'], api_key_required=True)
def replace_sled(pop, rack, rackunit, slot):
    rack = fix_rack(rack)
    rackunit = fix_rackunit(rackunit)
    refs = get_existing_refs(pop, rack, rackunit)
    if num_refs(pop, rack, rackunit, refs) == 0:
        return {'status': 'Error', 'message': 'No existing refs at {}/{}/{} Try create first, or fix by hand.'.format(pop, rack, rackunit)}
    bs_info = get_buildsheet(pop, rack, rackunit)
    if bs_info is None:
        return {'status': 'Error', 'message': 'No buildsheet records exist for {}/{}/{}.'.format(pop, rack, rackunit)}
    slot_info = slot_type(slot)
    if 'status' in slot_info and slot_info['status'] == 'Error':
        return slot_info

    # Loops through the buildsheet info to find the node we want
    nodes = [node for node in bs_info[slot_info['node_type']] if node['slot'] == int(slot_info['number'])]
    if len(nodes) == 0:
        return {'status': 'Error', 'error': 'No nodes found in buildsheet matching slot {} in {}/{}/{}'.format(slot, pop, rack, rackunit)}
    node = nodes[0]

    # Pulls the macaddr out of that node
    bs_macs = [macaddrs['address'] for macaddrs in node['mac_addresses'] if macaddrs['name'] == slot_info['bmc_name']]
    if len(bs_macs) == 0:
        return {'status': 'Error', 'error': 'No bmc macs found in buildsheet for {}/{}/{}/{}'.format(pop, rack, rackunit, slot)}
    bs_mac = bs_macs[0]

    # Finds the ref matching that node by ddns_hostname and our given params (ddns hostname is the ONLY reference to slot in infoblox)
    ddns = format_ddns_hostname(rack, rackunit, slot_info['dns_key'], slot_info['number'])
    ref_ar = [ref for ref in refs if ref['ddns_hostname'] == ddns]
    if len(ref_ar) == 0:
        # No node matching ddns_hostname in infoblox, try adding it as a new ref
        # Really should never ever hit this, but in case the fixedaddr gets removed from Infoblox, this allows rebuilding it
        network = get_infoblox_network(pop, rack)
        if network is None:
            return {'status': 'Error', 'error': 'Network object not found'}
        dhcprange = get_infoblox_dhcprange(network)
        if dhcprange is None:
            return {'status': 'Error', 'error': 'DHCPrange object not found'}
        extattr = set_extattr(pop, rack, rackunit)
        new_obj = entry(mac=bs_mac, ipv4addr=None, ddns_hostname=ddns, subnet=dhcprange, extattrs=extattr)
        # In case it had a lease already
        fix_preassigned(new_obj, network)
        print('Adding sled for empty ref at {}/{}/{}/{} with data: {}'.format(pop, rack, rackunit, slot, new_obj.toJson()))
        ref_ar = push_infoblox_list([new_obj])
        print('Created new ref during replace for {}/{}/{}/{}, Refs: {}'.format(pop, rack, rackunit, slot, ref_ar))
        if 'Error' in ref_ar[0]:
            return {'status': 'Fail', 'message': 'Creation of new ref for sled failed: {}'.format(refs)}
        return {'status': 'success', 'message': 'new ref created for sled: {}'.format(refs)}
    elif len(ref_ar) > 1:
        return {'status': 'Fail', 'error': 'More than 1 ref found matching ddns_hostname {}'.format_ddns_hostname(rack, rackunit, slot_info['dns_key'], slot_info['number'])}
    ref = ref_ar[0]
    ref_mac = ref['mac']
    if ref_mac is None:
        print("Error: No ref matched?")
        return {'status': 'Fail', 'message': 'No ref found matching {}/{}/{}/{}'.format(pop, rack, rackunit, slot)}
    if bs_mac != ref_mac:
        infoblox_data = {'mac': '{}'.format(bs_mac)}
        updated = update_ref(ref['_ref'], infoblox_data)
        return {'status': 'Updated', 'message': 'Sled at {}/{}/{}/{} mac {}, bsmac {}. Updated: {}'.format(pop, rack, rackunit, slot, ref_mac, bs_mac, updated)}
    return {'status': 'noop', 'message': 'Sled at {}/{}/{}/{} already has mac {} vs bs {}'.format(pop, rack, rackunit, slot, ref_mac, bs_mac)}


# Update a given ref with new data
def update_ref(ref, ib_data):
    username, password = getCredentials()
    # Update rw_action
    r = requests.put('https://{}/{}/{}'.format(ib_url, api_path, ref), data=ib_data, auth=(username, password), verify=False)
    resp = r.json()
    # resp = {'status': 'noop', 'message': 'Would have updated ref: {} with {}'.format(ref, ib_data)}
    print('Updated ref {} with data {}, resp: {}'.format(ref, ib_data, resp))
    return resp


@app.route('/restart', methods=['GET'], api_key_required=True)
def restart():
    restarted = restart_services()
    return restarted


# Pulls node data from buildsheet, puts into objects with network info
def get_nodes(pop, rack, rackunit, buildsheet, dhcprange, node_type='nodes'):
    objects = []
    # set for our ddns naming scheme (h for hubnode, s for normal node)
    if node_type == 'hub_nodes':
        dnskey = 'h'
    else:
        dnskey = 's'
    for node in buildsheet[node_type]:
        slot = node['slot']
        mac = get_mac(node, node_type)
        current_leases = active_leases(fetch_current_leases(mac))
        current_ip = addresses_from_leases(current_leases)
        ipv4 = None
        if len(current_ip) > 1:
            # If more than 1 address (lease), FAIL, this should never happen unless the rack subnet changes
            ipv4 = 'FAIL'
        elif len(current_ip) == 1:
            ipv4 = current_ip.pop()
        extattr = set_extattr(pop, rack, rackunit)
        obj = entry(mac=mac, ipv4addr=ipv4, ddns_hostname=format_ddns_hostname(rack, rackunit, dnskey, slot), subnet=dhcprange, extattrs=extattr)
        print("{} appended: {}/{}/{}/{}".format(node_type, pop, rack, rackunit, mac))
        objects.append(obj)
    return objects


# returns the mac address for a given buildsheet object
def get_mac(node, node_type):
    if node_type == 'hub_nodes':
        keyword = 'Base_MAC_ShMC'
    else:
        keyword = 'Base_MAC_BMC'
    for address in node['mac_addresses']:
        if address['name'] == keyword:
            mac = address['address']
    return mac


def get_infoblox_network(pop, rack):
    username, password = getCredentials()
    r = requests.get('https://{}/{}/network?*Site={}&*RackNumber={}&*NetDeviceType={}'.format(ib_url, api_path, pop, rack, 'ManagementAccess'), verify=False, auth=(username, password))
    if len(r.json()) != 1:
        print('got none or too many subnets, check extended attributes!')
        return None
    subnet = r.json()[0]['network']
    return subnet


def get_infoblox_dhcprange(subnet):
    username, password = getCredentials()
    r = requests.get('https://{}/{}/range?network={}'.format(ib_url, api_path, subnet), verify=False, auth=(username, password))
    if len(r.json()) != 1:
        print('got none or too many dhcpranges, check infoblox')
        return None
    resp = r.json()
    dhcprange = '{}-{}'.format(resp[0]['start_addr'], resp[0]['end_addr'])
    return dhcprange


def push_infoblox_list(objects):
    username, password = getCredentials()
    refs = []
    for item in objects:
        ref = push_infoblox_fixedaddress(username, password, item.toJson())
        refs.append(ref)
    return refs


def push_infoblox_fixedaddress(username, password, data):
    # Add rw_action
    r = requests.post('https://{}/{}/fixedaddress'.format(ib_url, api_path), verify=False, auth=(username, password), data=data)
    resp = r.json()
    # resp = {'status': 'noop', 'message': 'Would have pushed fixedaddress: {}'.format(data)}
    print('Adding fixedaddress for {}, resp: {}'.format(data, resp))
    return resp


def set_extattr(pop, rack, rackunit):
    if None in (pop, rack, rackunit):
        extattr = {'Site': {'value': 'Unknown'},
                   'RackNumber': {'value': 'Unknown'},
                   'Chassis': {'value': 'Unknown'}}
    else:
        extattr = {'Site': {'value': pop},
                   'RackNumber': {'value': rack},
                   'Chassis': {'value': "{:0>2}".format(rackunit)}}
    return extattr


# gets all IPs infoblox considers assignable
def get_available_space(network):
    username, password = getCredentials()
    free = []
    r = requests.get('https://{}/{}/ipv4address?network={}&lease_state=FREE'.format(ib_url, api_path, network), verify=False, auth=(username, password))
    for x in r.json():
        free.append(x)
    return len(free)


def restart_services():
    username, password = getCredentials()
    safe_restart, changes = check_pending_changes()
    if changes == 0:
        return {'status': 'Noop', 'error': 'There are no pending changes'}
    if not safe_restart:
        return {'status': 'Noop', 'error': 'There are pending changes owned by another user, verify in Infoblox GUI'}
    grid = get_infoblox_grid()
    if grid is None:
        return {"ERR": "issue finding grid object"}
    data = {"restart_option": "RESTART_IF_NEEDED", "services": "ALL", "user_name": username}
    # Restart rw_action
    r = requests.post('https://{}/{}/{}?_function=restartservices'.format(ib_url, api_path, grid), verify=False, auth=(username, password), data=data)
    resp = r.json()
    # resp = {'status': 'noop', 'message': 'Would have issued restart: {}'.format(data)}
    print('issued restart request to Infoblox: {}'.format(resp))
    return {'status': 'Unknown', 'message': 'Service restart requested', 'extended': resp}


def get_infoblox_grid():
    username, password = getCredentials()
    print('fetching grid from infoblox')
    r = requests.get('https://{}/{}/grid'.format(ib_url, api_path), verify=False, auth=(username, password))
    grid = r.json()[0]['_ref']
    if not grid:
        print('issue finding the grid.')
        return None
    return grid


def check_pending_changes():
    safe_restart = True
    changes = 0
    username, password = getCredentials()
    r = requests.get('https://{}/{}/grid:servicerestart:request:changedobject'.format(ib_url, api_path), verify=False, auth=(username, password))
    changes = len(r.json())
    for change in r.json():
        if change['user_name'] != username:
            print('Found changes by different user, not safe to restart')
            safe_restart = False
    return safe_restart, changes


def format_ddns_hostname(rack, rackunit, dnskey, slot):
    rack = fix_rack(rack)
    rackunit = fix_rackunit(rackunit)
    if dnskey in 'sh':
        return 'mgmt-{}.ru{:0>2}.{}{}'.format(rack, rackunit, dnskey, slot)
