#!/usr/bin/python -tt
'''
Script to convert leases in Infoblox to Fixedaddresses.

Pass in the following as args: pop, rack, rackunit, sled-identifier, mac
Example: $ python lease_to_fixedaddress.py sjc02 r404 13 s1 00:a0:a5:cc:b9:b6

Upon posting the fixedaddress object to Infoblox, an error will be returned if a given mac is
already in use by another fixedaddress object. I opted to not catch these errors but instead
just return them as the final output.

!!! The only line that will cause changes to Infoblox is the call to post_infoblox_fixedaddress
So for testruns (simply printing the object, I would comment out that call at line 242)

The script will print the object during a run, a proper object should look similar to the below
json structure.
(Note that the ipv4addr value may also be "func:nextavailableip:<dhcrangeStart-dhcprangeEnd>"
which means it will use the Infoblox API function call to fetch an ip from the dhcprange).

{
    "comment": "created by script",
    "ddns_hostname": "mgmt-r999.ru01.s1",
    "enable_ddns": true,
    "extattrs": {
        "Chassis": {
            "value": "1"
        },
        "RackNumber": {
            "value": "r999"
        },
        "Site": {
            "value": "pop01"
        }
    },
    "ipv4addr": "1.2.3.4",
    "mac": "00:12:34:56:78:90"
}

The final output of a successful run is the objects _ref value in Infoblox, example:
$ fixedaddress/ABCDEFGABCDEFGABCDEFGABCDEFGABCDEFG:1.2.3.4/default

This can be used to easily delete a created object via curl like:
$ curl -X DELETE -k -u adamnybe https://gm-infoblox.twitch.tv/wapi/v2.5/fixedaddress/ABCDEFGABCDEFGABCDEFGABCDEFGABCDEFG:1.2.3.4/default


Note that Infoblox services needs to be restarted for changes to happen, I would restart services
inbetween every new chassis/rackunit.
'''
import json
import os
import requests
import socket
import sys
from getpass import getpass
from requests.packages.urllib3.exceptions import InsecureRequestWarning  #noqa


# Setup infoblox base URI
ib_url = 'gm-infoblox.twitch.tv'


# Infoblox certificate...
requests.packages.urllib3.disable_warnings(InsecureRequestWarning)
# construct socket for ping_test
s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)


class fixedaddress(object):
    '''
    Fixedaddress class
    '''
    def __init__(self, comment='created by script', ipv4addr=None, mac=None, enable_ddns=True, ddns_hostname=None, subnet=None, extattrs=None):
        self.comment = comment  # differentiating the default comment from the lambda version
        self.mac = str(mac)
        self.enable_ddns = enable_ddns
        self.ipv4addr = str(ipv4addr)
        self.ddns_hostname = ddns_hostname
        self.extattrs = extattrs

    def toJson(self):
        # ensure we can get an iterable version of the object, could be done with __iter__ but imo this looks better
        return json.dumps(self, default=lambda o: o.__dict__, sort_keys=True, indent=4)


def get_credentials():
    '''
    Read credentials via userinput, password input is masked with getpass()
    For ease of use I recommend setting the environment variables or you will be prompted
    to input credentials on every invocation of the script.
    '''
    username = os.getenv('infoblox_username', raw_input('username: '))
    password = os.getenv('infoblox_password', getpass('password: '))
    return username, password


def fetch_current_leases(mac, usernane, password):
    '''
    Fetch all lease objects related to given mac address, construct a set() forcing uniqueness
    '''
    r = requests.get('https://{}/wapi/v2.5/lease?hardware={}'.format(ib_url, mac), verify=False, auth=(username, password))
    if len(r.json()) < 1:
        # no leases are found, fall back to using the func:nextavailableip:<dhcprange> function call
        print 'no lease found for {}, using func:nextavailableip'.format(mac)
        return None
    current = set()
    for lease in r.json():
        if lease['address'] not in current:
            # force uniqueness, no duplicates
            current.add(lease['address'])
    return current


def ping_test(addresses):
    '''
    In the case of multiple lease objects with different IP:s tied to the same mac we need to know which one to use.
    Perform a 'ping -c 1' to determine which address is responsive

    In the case that multiple addresses are responding, bail out as something may be very wrong in Infoblox.
    '''
    responding = set()
    for address in addresses:
        resp = os.system('ping -c 1 > /dev/null 2>&1' + ' ' + address)  # force no output
        if resp == 0 and address not in responding:
            responding.add(address)
    if len(responding) != 1:
        # none OR multiple addresses responding
        print 'none or multiple responding addresses found, check Infoblox lease objects for:'
        print addresses
        exit(1)
    return responding


def construct_extattrs(pop, rack, ru):
    '''
    Adds Extensible Attributes to our object
    '''
    extattr = {
               'Site': {'value': pop},
               'RackNumber': {'value': rack},
               'Chassis': {'value': str(ru)}
              }
    return extattr


def get_infoblox_network(pop, rack, username, password):
    '''
    Given a pop & rack, fetch the network/subnet assigned to the ManagementAccess devices in that rack

    If we get none or multiple subnets, bail out and let user verify Infoblox data
    '''
    r = requests.get('https://{}/wapi/v2.5/network?*Site={}&*RackNumber={}&*NetDeviceType={}'.format(ib_url, pop, rack, 'ManagementAccess'), verify=False, auth=(username, password))
    if len(r.json()) is not 1:
        # rack has none OR more than one management network
        print 'got none or too many subnets, check extended attributes in infoblox for {} {}'.format(pop, rack)
        exit(1)
    subnet = r.json()[0]['network']
    return subnet


def get_infoblox_dhcprange(subnet, username, password):
    '''
    Fetch the dhcprange of a given network/subnet, to ensure we don't blindly allocate from the start
    '''
    r = requests.get('https://{}/wapi/v2.5/range?network={}'.format(ib_url, subnet), verify=False, auth=(username, password))
    if len(r.json()) is not 1:
        # multiple dhcpranges for a single network is not something we want to deal with right now.
        print 'found none or too many dhcpranges, check infoblox'
        exit(1)
    resp = r.json()
    dhcprange = '{}-{}'.format(resp[0]['start_addr'], resp[0]['end_addr'])
    return dhcprange


def post_infoblox_fixedaddress(fixedaddress, username, password):
    '''
    Post fixedaddress object to Infoblox
    This will fail if the mac address is already in use by another fixedaddress object
    I opted to not implement catching any infoblox-errors and instead have them returned to the caller
    '''
    r = requests.post('https://{}/wapi/v2.5/fixedaddress'.format(ib_url), verify=False, auth=(username, password), data=fixedaddress)
    resp = r.json()
    return resp


if __name__ == '__main__':
    '''
    Pass in the following as args: pop, rack, rackunit, sled-identifier, mac
    Example: $ python lease_to_fixedaddress.py sjc02 r404 13 s1 00:a0:a5:cc:b9:b6
    '''
    if len(sys.argv) < 6:
        print 'not enough args given.'
        print 'Required args: pop, rack, rackunit, sled-identifier, mac'
        print 'Example: $ python lease_to_fixedaddress.py sjc02 r404 13 s1 00:a0:a5:cc:b9:b6'
        exit(1)
    # read arguments
    pop = sys.argv[1]
    rack = sys.argv[2].lower()  # force lowercase
    ru = sys.argv[3]
    sled_type = sys.argv[4].lower()  # force lowercase
    mac = sys.argv[5]

    # get_credentials (from env or via input)
    username, password = get_credentials()

    # pad rackunit
    rackunit = "{:0>2}".format(ru)

    # construct extattrs
    extended = construct_extattrs(pop, rack, rackunit)

    # fetch current leases
    current = fetch_current_leases(mac, username, password)  # also filters out addresses
    if current is None:
        # no addresses found means we fall back to using func:nextavailableip, so we need the network and dhcprange
        subnet = get_infoblox_network(pop, rack, username, password)  # fetch network/subnet for pop+rack
        dhcprange = get_infoblox_dhcprange(subnet, username, password)  # fetch dhcprange for network/subnet
        ipv4address = 'func:nextavailableip:{}'.format(dhcprange)
    else:
        # addresses were found, we test responsiveness via ping_test to determine which one to use
        responsive = ping_test(current)
        ipv4address = responsive.pop()  # get address from set

    # construct fixedaddress object (the class will force necessary things to be strings)
    entry = fixedaddress(ipv4addr=ipv4address, mac=mac, ddns_hostname='mgmt-{}.ru{}.{}'.format(rack, rackunit, sled_type), extattrs=extended)

    # print object
    print entry.toJson()

    # force an iterable version of object via toJson() method
    data = entry.toJson()

    # Post fixedaddress to Infoblox
    # !!! This is the only line which will cause changes to Infoblox, comment it out for testruns
    #
    output = post_infoblox_fixedaddress(data, username, password)

    # print returned _ref of object OR any Infoblox error
    if output:
        print output
