package main

import (
	"crypto/tls"
	"encoding/json"
	"io"
	"log"
	"net/http"
	"os"

	infoblox "code.justin.tv/systems/go-infoblox"
	"github.com/gorilla/mux"
)

var (
	client *infoblox.Client
)

// getEnvVar, helper to let us assign defaultvalues when something is missing from env
// mimic behaviour of pythons os.getenv
func getEnvVar(key, defaultvalue string) string {
	if value, exists := os.LookupEnv(key); exists {
		return value
	}
	return defaultvalue
}

// SetupClient function
func SetupClient() *infoblox.Client {
	config := infoblox.DefaultConfig()
	// XXX: This looks messy, separate it?
	config.HTTPClient = &http.Client{
		Transport: &http.Transport{
			TLSClientConfig: &tls.Config{
				InsecureSkipVerify: true,
			},
		},
	}
	// Opt for latest available version of wapi
	config.Endpoint = getEnvVar("LEASER_ENDPOINT", "https://gm-infoblox.twitch.tv/wapi/v2.5")
	// XXX: Read defaults from config file instead?
	config.Username = getEnvVar("LEASER_USERNAME", "")
	config.Password = getEnvVar("LEASER_PASSWORD", "")

	client, err := infoblox.NewClient(config)
	if err != nil {
		log.Fatalf("Client setup failed\n")
	}
	return client
}

// GetLease will fetch a lease based on an ip sent in the url
// For example a GET of /lease/<ip> will return either an empty array (no result found)
// or a json representation of the fixedaddress object
func GetLease(w http.ResponseWriter, r *http.Request) {
	params := mux.Vars(r)
	// Has to be slice of infoblox.Fixedaddress because infoblox always returns an array
	lease := []infoblox.Fixedaddress{}
	_, err := client.GetLease(params["ip"], &lease)
	if err != nil {
		log.Printf("Could not find lease for %v\n", params["ip"])
	}
	log.Printf("Response: %#v\n", lease)

	responsejson, err := json.Marshal(lease)
	if err != nil {
		log.Printf("json marshaling failed for response %v\n", lease)
		w.WriteHeader(http.StatusInternalServerError)
		io.WriteString(w, `[]`)
		return
	}
	w.WriteHeader(http.StatusOK)
	io.WriteString(w, string(responsejson))
}

// GetLeaseByMac will, given a mac address, fetch a lease object from infoblox
// For example a GET of /mac/<mac will return either an empty array (no result found)
// or a json representation of the fixedaddress object from infoblox
func GetLeaseByMac(w http.ResponseWriter, r *http.Request) {
	params := mux.Vars(r)
	lease := []infoblox.Fixedaddress{}
	_, err := client.GetLeaseByMac(params["mac"], &lease)
	if err != nil {
		log.Printf("Could not find lease for %v\n", params["mac"])
	}
	log.Printf("Response: %#v\n", lease)

	responsejson, err := json.Marshal(lease)
	if err != nil {
		log.Printf("json marshaling failed for response %v\n", lease)
		w.WriteHeader(http.StatusInternalServerError)
		io.WriteString(w, `[]`)
		return
	}
	w.WriteHeader(http.StatusOK)
	io.WriteString(w, string(responsejson))
}

// CreateLease will post a lease to infoblox via the wapi
// returns either "success": true or "success": false
func CreateLease(w http.ResponseWriter, r *http.Request) {
	lease := infoblox.Fixedaddress{}
	_ = json.NewDecoder(r.Body).Decode(&lease)
	resp, err := client.PostLease(&lease)
	if err != nil {
		log.Printf("Could not create lease for %v\n", &lease.IPV4Address)
		w.WriteHeader(http.StatusInternalServerError)
		io.WriteString(w, `{"success": false}`)
		return
	}
	log.Printf("Response: %v\n", resp)

	w.WriteHeader(http.StatusCreated)
	io.WriteString(w, `{"success": true}`)
}

// DeleteLease takes a reference passed with the variables of the request
// returns either "success": true or "success": false
// Since we can delete any object based on reference we need to validate target is Fixedaddress
// Regexp validation happens in the handler, bad requests should not reach function
// Infoblox references for Fixedaddress objects _should_ always follow this:
// fixedaddress/[A-Za-z0-9]+:[0-9.]+/default
func DeleteLease(w http.ResponseWriter, r *http.Request) {
	params := mux.Vars(r)
	// XXX: Some of the following logic may be superfluous with regexp matching in handler
	if ref, exists := params["ref"]; !exists {
		log.Printf("Bad request, reference missing.\n")
		w.WriteHeader(http.StatusBadRequest)
		io.WriteString(w, `{"success": false}`)
		return
	} else if ref == "" {
		log.Printf("Bad request, reference is empty/lost.\n")
		w.WriteHeader(http.StatusBadRequest)
		io.WriteString(w, `{"success": false}`)
		return
	} else {
		resp, err := client.Delete(ref)
		if err != nil {
			log.Printf("Delete request failed for %v\n", ref)
			w.WriteHeader(http.StatusInternalServerError)
			io.WriteString(w, `{"success": false}`)
		}
		log.Printf("Response: %v\n", resp)

		w.WriteHeader(http.StatusOK)
		io.WriteString(w, `{"success": true}`)
		return
	}
}

func main() {
	client = SetupClient()
	router := mux.NewRouter()
	router.HandleFunc("/lease/{ip}", GetLease).Methods("GET")
	router.HandleFunc("/lease", CreateLease).Methods("POST")
	router.HandleFunc("/mac/{mac}", GetLeaseByMac).Methods("GET")
	router.HandleFunc("/delete/{ref:fixedaddress/[A-Za-z0-9]+:[0-9.]+/default}", DeleteLease).Methods("DELETE")
	log.Fatal(http.ListenAndServe(":8000", router))
}
