package main

import (
	"encoding/json"
	"errors"
	"flag"
	"fmt"
	"io/ioutil"
	"log"
	"os"
	"os/exec"
	"path/filepath"
	"reflect"
)

const tagName = "fpm"

// FpmManifest describes the fpm manifest file
type FpmManifest struct {
	URL                     string `fpm:"url" json:"url"`
	Name                    string `fpm:"name" json:"name"`
	Type                    string `fpm:"output-type" json:"type"`
	License                 string `fpm:"license" json:"license"`
	Version                 string `fpm:"version" json:"version"`
	Iteration               string `fpm:"iteration" json:"iteration"`
	Maintainer              string `fpm:"maintainer" json:"maintainer"`
	RpmSummary              string `fpm:"rpm-summary" json:"rpm-summary"`
	Description             string `fpm:"description" json:"description"`
	Architecture            string `fpm:"architecture" json:"architecture"`
	DebChangelog            string `fpm:"deb-changelog" json:"deb-changelog"`
	RpmChangelog            string `fpm:"rpm-changelog" json:"rpm-changelog"`
	DebNoDefaultConfigFiles string `fpm:"deb-no-default-config-files" json:"deb-no-default-config-files"`

	DebPreDepends []string `fpm:"deb-pre-depends" json:"deb-pre-depends"`
	Depends       []string `fpm:"depends" json:"depends"`
	Provides      []string `fpm:"provides" json:"provides"`
	Conflicts     []string `fpm:"conflicts" json:"conflicts"`
	Replaces      []string `fpm:"replaces" json:"replaces"`

	BeforeInstall string `fpm:"before-install" json:"before-install"`
	AfterInstall  string `fpm:"after-install" json:"after-install"`

	BeforeRemove string `fpm:"before-remove" json:"before-remove"`
	AfterRemove  string `fpm:"after-remove" json:"after-remove"`

	BeforeUpgrade string `fpm:"before-upgrade" json:"before-upgrade"`
	AfterUpgrade  string `fpm:"after-upgrade" json:"after-upgrade"`
}

func (manifest *FpmManifest) getArgs() ([]string, error) {
	var args []string

	v := reflect.ValueOf(manifest).Elem()
	for i := 0; i < v.NumField(); i++ {
		// fpm tag is used as fpm argument structuring
		tag := fmt.Sprintf("--%s", v.Type().Field(i).Tag.Get(tagName))

		// assert field type for unpacking slices into fpm arguments
		switch t := v.Field(i).Interface().(type) {
		case string:
			if t == "" {
				continue
			}
			args = append(args, tag, t)
		case []string:
			if len(t) > 0 {
				for _, x := range t {
					args = append(args, tag, x)
				}
			}
		default:
			// there really shouldn't be any other type of field
			return nil, errors.New("I would panic here, type unknown")
		}
	}
	return args, nil
}

// FPMfromDir runs `fpm` from a target directory, and returns the combined stdout/err, and
// an error if there was one.
func (manifest *FpmManifest) FPMfromDir(dir string) ([]byte, error) {
	args, err := manifest.getArgs()
	if err != nil {
		return nil, err
	}
	cmd := exec.Command("fpm", args...)
	cmd.Args = append(cmd.Args, "-s", "dir", "-C", "fpm_source")
	cmd.Dir = dir
	return cmd.CombinedOutput()
}

// FPM runs `fpm` from the current directory, and returns stdout/err and an error if there is one.
func (manifest *FpmManifest) FPM() ([]byte, error) {
	return manifest.FPMfromDir("")
}

func main() {
	log.SetFlags(0)
	flag.Usage = func() {
		base := filepath.Base(os.Args[0])
		fmt.Fprintf(os.Stderr, "Usage of %s:\n\t%s FILE\n", base, base)
	}
	flag.Parse()

	if len(flag.Args()) == 0 {
		log.Fatal("No Arguments given, need filename.")
	}

	filename := flag.Arg(0)

	file_contents, err := ioutil.ReadFile(filename)
	if err != nil {
		log.Fatal(err)
	}

	manifest := FpmManifest{}
	if err := json.Unmarshal(file_contents, &manifest); err != nil {
		log.Fatal(err)
	}

	b, err := manifest.FPMfromDir(filepath.Dir(filename))
	if err != nil {
		log.Fatal(string(b), err)
	}
	fmt.Println(string(b))
}
