package main

import (
	"encoding/json"
	"fmt"
	"testing"
)

const argLength = 42

const jsonData = `
{
  "architecture": "amd64",
  "name": "video-haproxy",
  "maintainer": "video-coreservices",
  "version": "1.9.9",
  "url": "https://git.xarth.tv/video-coreservices",
  "type": "deb",
  "description": "fast and reliable load balancing reverse proxy",
  "depends": [
    "libssl1.0.0 (>= 1.0.2)",
    "libpcre3",
    "zlib1g",
    "netcat"
  ],
  "rpm-summary": "hello world",
  "deb-changelog": "hello world",
  "rpm-changelog": "hello world",
  "deb-pre-depends": [
    "foo",
    "bar"
  ],
  "deb-no-default-config-files": "nope",
  "provides": [
    "haproxy"
  ],
  "conflicts": [
    "haproxy"
  ],
  "replaces": [
    "httpd"
  ],
  "after-remove": "testing.sh"
}
`

func TestManifestArgs(t *testing.T) {
	args, err := getManifestArgs()
	if err != nil {
		t.Error(err)
	}
	if len(args) != argLength {
		t.Errorf("Args length is %d, expected %d", len(args), argLength)
	}
}

func TestDependsList(t *testing.T) {
	args, err := getManifestArgs()
	if err != nil {
		t.Error(err)
	}

	count := countArgs("depends", args)
	if count != 4 {
		t.Errorf("Wrong number of depends arguments %d, expected 4", count)
	}
}

func TestDebPreDependsList(t *testing.T) {
	args, err := getManifestArgs()
	if err != nil {
		t.Error(err)
	}

	count := countArgs("deb-pre-depends", args)
	if count != 2 {
		t.Errorf("Wrong number of depends arguments %d, expected 4", count)
	}
}

func TestConflictsList(t *testing.T) {
	args, err := getManifestArgs()
	if err != nil {
		t.Error(err)
	}

	count := countArgs("conflicts", args)
	if count != 1 {
		t.Errorf("Wrong number of conflicts arguments %d, expected 1", count)
	}
}

func TestReplacesList(t *testing.T) {
	args, err := getManifestArgs()
	if err != nil {
		t.Error(err)
	}

	count := countArgs("replaces", args)
	if count != 1 {
		t.Errorf("Wrong number of replaces arguments %d, expected 1", count)
	}
}

func TestProvidesList(t *testing.T) {
	args, err := getManifestArgs()
	if err != nil {
		t.Error(err)
	}

	count := countArgs("provides", args)
	if count != 1 {
		t.Errorf("Wrong number of provides arguments %d, expected 1", count)
	}
}

func countArgs(arg string, args []string) (count int) {
	for _, s := range args {
		if s == fmt.Sprintf("--%s", arg) {
			count++
		}
	}
	return
}

func getManifestArgs() ([]string, error) {
	manifest := FpmManifest{}
	if err := json.Unmarshal([]byte(jsonData), &manifest); err != nil {
		return nil, err
	}
	return manifest.getArgs()
}
