node default {

  Exec { path => [ '/bin', '/sbin', '/usr/bin', '/usr/sbin', '/usr/local/bin', '/opt/puppetlabs/bin'] }

  $twitch_fqdn = "${::twitch_hostname}.${::domain}"

  class {'::finish': } -> class {'::checkin': }

}

# lint:ignore:autoloader_layout
# teamd_conf($team_slaves) - generate teamd.conf with $team_slaves array
define teamd_conf($team_slaves) {

  $teamd_conf_content = @("END")
    <%-
      port = {"prio" => -100, "sticky" => false}
      d = {
        "device"=>"team0",
        "hwaddr"=>"${::twitch_macaddress}",
          "runner"=>{
            "name"=>"lacp",
            "active"=>true,
            "fast_rate"=>true,
            "tx_hash"=>["l3", "l4"]
          },
          "link_watch"=>{
            "name"=>"ethtool"
          },
          "ports" => Hash[@team_slaves.map {|k| [k, port]}]
      }
    -%>
    <%= JSON.dump(d) %>
    | END

  $teamd_systemd_unit_file_content = @("END")
    [Unit]
    Description=Team Daemon for device team0
    Before=network-pre.target
    Wants=network-pre.target

    [Service]
    ExecStart=/usr/bin/teamd -f /etc/teamd.conf -o -r
    ExecStop=/usr/bin/teamd -f /etc/teamd.conf -k
    Restart=always

    [Install]
    WantedBy=multi-user.target
    | END

  file { '/etc/teamd.conf':
    ensure  => present,
    owner   => 'root',
    group   => 'root',
    mode    => '0644',
    content => inline_template($teamd_conf_content),
  }

  file { '/etc/systemd/system/teamd.service':
    ensure  => present,
    owner   => 'root',
    group   => 'root',
    mode    => '0644',
    content => $teamd_systemd_unit_file_content,
  }

  service { 'teamd.service':
    ensure  => false,
    enable  => true,
    require => File['/etc/systemd/system/teamd.service'],
  }

}

# systemd_link() - configure interface name persistence
define systemd_link (
  $driver,
  $ensure = 'absent',
) {

  $systemd_link   = "/etc/systemd/network/00-${name}.link"

  $systemd_link_contents = @("END")
    [Match]
    Driver=${driver}
    MACAddress=<%= scope["macaddress_#{@name}"] %>

    [Link]
    Name=${name}
    | END

  file { $systemd_link:
    ensure  => $ensure,
    owner   => 'root',
    group   => 'root',
    mode    => '0644',
    content => inline_template($systemd_link_contents),
  }

}

# systemd_module_load() - configure module to preload via systemd
define systemd_module_load() {

  case $name {
    'i40e': { $ensure_module_load = 'file' }
    'mlx5_core': { $ensure_module_load = 'file' }
    default: { $ensure_module_load = 'absent' }
  }

  file { "/etc/modules-load.d/${name}.conf":
    ensure  => $ensure_module_load,
    owner   => 'root',
    group   => 'root',
    mode    => '0644',
    content => "${name}\n",
  }

  $systemd_link_for_interfaces = split(inline_template('<%= scope["interfaces_#{@name}"] %>'), ',')

  systemd_link { $systemd_link_for_interfaces:
    ensure => $ensure_module_load,
    driver => $name,
  }

}

# team_iface_conf() - create interfaces file for team interface
define team_iface_conf() {
  $interface_conf = "/etc/network/interfaces.d/${name}.cfg"

  $interface_conf_contents = @("END")
    auto ${name}
    iface ${name} inet dhcp
    | END

  file { $interface_conf:
    ensure  => present,
    owner   => 'root',
    group   => 'root',
    mode    => '0644',
    content => $interface_conf_contents,
  }

}

# finish - configure system
class finish {
  # List of machine classes that by default will set the node-enabled healthcheck to fail.
  $consul_disable_machine_classes = [
    'video-weaver',
    'video-weaver-lb',
    'frontier',
    'video-tier1',
    'ingest-proxy-lb',
    's3-http-proxy',
  ]
  $proxy_host     = 'proxy.internal.justin.tv:9797/'

  $ssh_key = 'AAAAB3NzaC1yc2EAAAADAQABAAACAQCyi9cp2+n0d3gfwpfARCIsie0CIpOM2L9UyZwpcdaNOwJ/cHwca1svxg0AMjMK5cG88W37n7MeM7LJ1M+i8TO8WqNymg1pGJMMGTJgLlUHH7HB6Fvxh3yqLvq4J1tA8MX1eM4cKq3HfOWgCU9Kw3E0Wcr4Ss9C38SgMNXLq0L66HZRd4yF737sSTpVuC70iT9cGOoRjfzEFmFy+HbUaJ2VveTBBBSv0f5wdHF3JbGkmgUmNCMVf4hZLIUPxv6jLWDUPpl7fqQwxjy5rg+l9pGqpHAN46RIVr6XwX/YhMWN7bWCDNMB+dupSFLvHncwO9DwFzifk7aVWtTOit21O0yYWzCWfipUxJP6ybc1Hv6Z7uljgIo4AU7/VCY/T48cIwvRPbuGXVqo2yWi3mU/gpXhCvIAE2oTFljetFvMnbIm8OWg7UIid8CGTkZMrsTrwMjBn+HdeLYDhMqegmNcEtCvUOHJhMd2n/hzluX4jaczZdpPk4lMmlPHgCBNtphlgIlvlVeVvGD4jwrtrDD79cbifc4RW8cfOz45DhyT2HhqsbO+X6HAjejOuDItGA9ve4C7R9CZ0JXqcMtw4EKRNbhfVLeIY9spb1WOrSxzsotTxabDbB7LNyZJ2aVTnQCVjb4QUo2lySVXXuRpnuEbqapVn0DvSfQB2hdewib8sULOow==' # lint:ignore:140chars

  ssh_authorized_key { 'jtv':
    user => 'jtv',
    type => 'ssh-rsa',
    key  => $ssh_key,
  }

  file { '/etc/sudoers.d/jtv':
    owner   => 'root',
    group   => 'root',
    mode    => '0440',
    content => "jtv ALL=NOPASSWD: ALL\n",
  }

  file {'/etc/default/twitch_tags':
    owner   => 'root',
    group   => 'root',
    mode    => '0444',
    content => "POP=${::pop}\nSERVICE=${::machine_class}\nENVIRONMENT=${::environment}\nOWNER=${::twitch_team}\n",
  }

  file { '/root/.aws':
    ensure => directory,
    owner  => 'root',
    group  => 'root',
    mode   => '0700',
  }

  $sssd_conf_contents = @("END")
    # File managed by Puppet
    [sssd]
    config_file_version = 2
    domains = justin.tv
    services = nss,pam,ssh

    [nss]
    filter_groups = ${::local_groups}
    filter_users = ${::local_users}

    [domain/justin.tv]
    auth_provider = ldap
    cache_credentials = true
    debug_level = 0
    id_provider = ldap
    ignore_group_members = true
    ldap_backup_uri = ldaps://ldap-vip.internal.justin.tv
    ldap_group_member = uniqueMember
    ldap_group_nesting_level = 0
    ldap_group_search_base = ou=Groups,dc=justin,dc=tv?onelevel?
    ldap_id_use_start_tls = true
    ldap_schema = rfc2307bis
    ldap_search_base = dc=justin,dc=tv
    ldap_tls_cacert = /etc/ssl/certs/ca-certificates.crt
    ldap_tls_reqcert = demand
    ldap_uri = ldaps://ldap-ro.internal.justin.tv
    ldap_user_search_base = ou=Users,dc=justin,dc=tv?onelevel?
    min_id = 10000
    refresh_expired_interval = 4050
    sudo_provider = none
    | END

  $sssd_ensure = $::lsbdistcodename ? {
    'bionic' => 'present',
    default  => 'absent',
  }

  file { '/etc/sssd/sssd.conf':
    ensure  => $sssd_ensure,
    owner   => 'root',
    group   => 'root',
    mode    => '0600',
    content => $sssd_conf_contents,
  }

  # lint:ignore:140chars
  $proxy_sh_contents = @("END"/$)
  http_proxy='http://${proxy_host}'
  HTTP_PROXY='http://${proxy_host}'
  https_proxy='http://${proxy_host}'
  HTTPS_PROXY='http://${proxy_host}'
  ftp_proxy='http://${proxy_host}'
  FTP_PROXY='http://${proxy_host}'
  no_proxy="169.254.169.254,localhost,127.0.0.1,127.0.1.1,127.0.0.0/8,199.9.248.0/21,192.16.64.0/21,10.0.0.0/8,.internal.justin.tv,.internal.twitch.tv,.justin.tv,.twitch.tv,.jtvnw.net,packages.internal.justin.tv,pypi.internal.justin.tv,package-repo.internal.justin.tv,.live-video.a2z.com,.twitch.a2z.com"
  NO_PROXY="169.254.169.254,localhost,127.0.0.1,127.0.1.1,127.0.0.0/8,199.9.248.0/21,192.16.64.0/21,10.0.0.0/8,.internal.justin.tv,.internal.twitch.tv,.justin.tv,.twitch.tv,.jtvnw.net,packages.internal.justin.tv,pypi.internal.justin.tv,package-repo.internal.justin.tv,.live-video.a2z.com,.twitch.a2z.com"
  export http_proxy HTTP_PROXY https_proxy HTTPS_PROXY ftp_proxy FTP_PROXY no_proxy NO_PROXY
  | END
  # lint:endignore

  $ensure_proxy_file = $::public_ip ? {
    'true'  => 'absent',
    default => 'present',
  }

  file { '/etc/default/twitch_proxy':
    ensure  => $ensure_proxy_file,
    owner   => 'root',
    group   => 'root',
    mode    => '0444',
    content => "HTTP_PROXY=${proxy_host}\nhttp_proxy=${proxy_host}",
  }

  file { '/etc/profile.d/proxy.sh':
    ensure  => $ensure_proxy_file,
    owner   => 'root',
    group   => 'root',
    mode    => '0644',
    content => $proxy_sh_contents,
  }

  $use_public_ip = $::public_ip ? {
    'true'  => '',
    default => '#',
  }

  $dhclient_host_name = $::lsbdistcodename ? {
    precise => '"<hostname>"',
    default => '= gethostname()',
  }

  $dhclient_conf_contents = @("END")
    option rfc3442-classless-static-routes code 121 = array of unsigned integer 8;

    send host-name ${dhclient_host_name};
    request subnet-mask, broadcast-address, time-offset, routers,
    	       domain-name, domain-name-servers, domain-search, host-name,
    	       netbios-name-servers, netbios-scope, interface-mtu,
    	       rfc3442-classless-static-routes, ntp-servers,
    	       dhcp6.domain-search, dhcp6.fqdn,
    	       dhcp6.name-servers, dhcp6.sntp-servers;

    option resolv-options code 224 = text;
    also request resolv-options;
    ${use_public_ip}send user-class "PUBLIC";
    | END

  file { '/etc/dhcp/dhclient.conf':
    owner   => 'root',
    group   => 'root',
    mode    => '0644',
    content => $dhclient_conf_contents,
  }

  $netplan_conf_ensure = $::lsbdistcodename ? {
    'bionic' => 'present',
    default  => 'absent',
  }

  $netplan_conf_contents = @(END)
    network:
      version: 2
      renderer: networkd
    | END

  file { '/etc/netplan/01-netcfg.yaml':
    ensure  => $netplan_conf_ensure,
    owner   => 'root',
    group   => 'root',
    mode    => '0644',
    content => $netplan_conf_contents,
  }

  $resolv_options_contents = @(END)
    if [ "$new_resolv_options" ]; then
        echo "options $new_resolv_options" >> /etc/resolv.conf
    fi
    | END

  file { '/etc/dhcp/dhclient-exit-hooks.d/resolvoptions':
    owner   => 'root',
    group   => 'root',
    mode    => '0644',
    content => $resolv_options_contents,
  }

  if $::puppet_environment == 'puppet5' {
    $puppet_conf_contents = @("END")
      [main]
      ca_server       = ${::puppet_ca_server}
      server          = ${::puppet_server}
      logdir          = /var/log/puppetlabs
      stringify_facts = false

      [agent]
      server      = ${::puppet_server}
      environment = ${::puppet_environment}
      runinterval = 1800
      splay       = false
      daemonize   = false
      | END
  } else {
    $puppet_conf_contents = @("END")
      [agent]
      pluginsync      = true
      report          = true
      ignoreschedules = true
      daemonize       = false
      ca_server       = ${::puppet_ca_server}
      environment     = production
      server          = ${::puppet_server}
      [main]
      trusted_node_data = true
      stringify_facts = false
      | END
  }

  file { '/etc/puppet/puppet.conf':
    ensure  => present,
    owner   => 'root',
    group   => 'root',
    mode    => '0644',
    content => $puppet_conf_contents,
  }

  file { '/opt/omnibus-puppet/etc/puppet.conf':
    ensure => absent,
  }

  $facter_dirs = ['/etc/facter', '/etc/facter/facts.d/']

  file { $facter_dirs:
    ensure => directory,
    owner  => 'root',
    group  => 'root',
    mode   => '0644',
  }

  file { '/etc/facter/facts.d/clean.txt':
    ensure  => 'present',
    owner   => 'root',
    group   => 'root',
    mode    => '0644',
    content => "clean=true\n",
  }

  file { '/etc/facter/facts.d/provisioner_server.txt':
    ensure  => present,
    owner   => 'root',
    group   => 'root',
    mode    => '0644',
    content => "provisioner_server=${::provisioner_server}\n",
  }

  file { '/etc/facter/facts.d/twitch_team.txt':
    owner   => 'root',
    group   => 'root',
    mode    => '0644',
    content => "twitch_team=${::twitch_team}\n",
  }

  file { '/etc/facter/facts.d/twitch_environment.txt':
    ensure  => $::mck_file_ensure,
    owner   => 'root',
    group   => 'root',
    mode    => '0644',
    content => "twitch_environment=${environment}\n",
  }

  file { '/etc/facter/facts.d/twitch_role.txt':
    ensure  => $::mck_file_ensure,
    owner   => 'root',
    group   => 'root',
    mode    => '0644',
    content => "twitch_role=${::twitch_role}\n",
  }

  file { '/etc/facter/facts.d/machine_class.txt':
    owner   => 'root',
    group   => 'root',
    mode    => '0644',
    content => "machine_class=${::machine_class}\n",
  }

  if $::disable_anycast == 'true' {

    case $::twitch_hostname {
      /^(pop-master.*|dns-(auth|cache|resolver|hidden)|ldap|mail|postfix|squid-proxy)-[a-f0-9]{6}/: { $disable_anycast_fact = present }
      default: { $disable_anycast_fact = absent }
    }

    file { '/etc/facter/facts.d/disable_anycast.txt':
      ensure  => $disable_anycast_fact,
      owner   => 'root',
      group   => 'root',
      mode    => '0644',
      content => "disable_anycast=true\n",
    }

  }

  if $::machine_class in $consul_disable_machine_classes{
      $partial_consul      = 'file'
      $disable_consul_fact = 'absent'
  } else {
      $partial_consul      = 'absent'
      $disable_consul_fact = 'file'
  }

  file { '/etc/consul':
    ensure => directory,
    owner  => 'root',
    group  => 'root',
    mode   => '0755',
  }

  file { '/etc/consul/consul_disabled':
    ensure  => $partial_consul,
    owner   => 'root',
    group   => 'root',
    mode    => '0644',
    content => "Consul is partially disabled on newly provisioned hosts.\n",
  }

  if $::disable_consul == 'true' {
    file { '/etc/facter/facts.d/disable_consul.txt':
      ensure  => $disable_consul_fact,
      owner   => 'root',
      group   => 'root',
      mode    => '0644',
      content => "disable_consul=true\n",
    }
  }

  $firstboot_crond_contents = @("END")
    PUPPET_CA_SERVER=${::puppet_ca_server}
    PUPPET_SERVER=${::puppet_server}
    PUPPET_ENVIRONMENT=${::puppet_environment}
    RUN_PUPPET=${::run_puppet}

    @reboot root /opt/twitch/provision-files/firstboot.sh 2>&1 | tee -a /var/log/firstboot.log | logger -t firstboot
    | END

  file { '/etc/cron.d/firstboot':
    ensure  => present,
    owner   => 'root',
    group   => 'root',
    mode    => '0644',
    content => $firstboot_crond_contents,
  }

  file { '/etc/hostname':
    ensure  => present,
    owner   => 'root',
    group   => 'root',
    mode    => '0644',
    content => "${::twitch_hostname}\n",
  }

  if ( $::fqdn != $::twitch_fqdn ) {

    host {$::fqdn:
      ensure       => absent,
      ip           => '127.0.1.1',
      host_aliases => $::hostname,
    }

  }

  host {'localhost':
    ensure => present,
    ip     => '127.0.0.1',
  }
  host { $twitch_fqdn: # lint:ignore:variable_scope
    ensure       => present,
    ip           => '127.0.1.1',
    host_aliases => $::twitch_hostname,
  }
  host {'ip6-localhost':
    ensure       => present,
    ip           => '::1',
    host_aliases => 'ip6-loopback',
  }
  host {'ip6-localnet':
    ensure => present,
    ip     => 'fe00::0',
  }
  host {'ip6-mcastprefix':
    ensure => present,
    ip     => 'ff00::0',
  }
  host {'ip6-allnodes':
    ensure => present,
    ip     => 'ff02::1',
  }
  host {'ip6-allrouters':
    ensure => present,
    ip     => 'ff02::2',
  }

  file { '/etc/network/interfaces.d':
    ensure => directory,
    owner  => 'root',
    group  => 'root',
    mode   => '0755',
  }

  $interfaces_content = @(END)
    #####################
    # Managed by Puppet #
    #####################
    # This file describes the network interfaces available on your system
    # and how to activate them. For more information, see interfaces(5).

    # The loopback network interface
    auto lo
    iface lo inet loopback

    # Source interfaces
    # Please check /etc/network/interfaces.d before changing this file
    # as interfaces may have been defined in /etc/network/interfaces.d
    # NOTE: the primary ethernet device is defined in
    # /etc/network/interfaces.d/eth0
    # See LP: #1262951
    source /etc/network/interfaces.d/*.cfg
    | END

  file { '/etc/network/interfaces':
    ensure  => present,
    owner   => 'root',
    group   => 'root',
    mode    => '0644',
    content => $interfaces_content,
  }

  $module_load = split($::network_drivers, ',')

  systemd_module_load { $module_load:
  }

  $interfaces_prep = split($::interfaces_with_link,',')

  team_iface_conf { 'team0':
    require => File['/etc/network/interfaces.d'],
  }

  teamd_conf { 'teamd.conf':
    team_slaves => $interfaces_prep,
  }

  service { 'puppet':
    ensure => stopped,
    enable => false,
  }

  file {'/lib/systemd/systemd-resolved':
    ensure => present,
    owner  => 'root',
    group  => 'root',
    mode   => '0644',
  }

  service { 'systemd-resolved':
    ensure => stopped,
    enable => false,
  }

}

# checkin - notify provisioner this script is done. set do_not_set_hostname qs
# overwise provisioner will override the hostname everytime.
class checkin {

  exec { 'checkin':
    command => "wget --no-hsts --tries=10 --no-proxy -O /dev/null -q 'http://${::provisioner_server}/checkin/${::twitch_macaddress}/${twitch_fqdn}?state=preseed%20finish%20script%20done&final_state=1&do_not_set_hostname=1'", # lint:ignore:variable_scope
  }

  package { 'resolvconf':
    ensure  => absent,
    require => Exec['checkin'],
  }

}
# lint:endignore
