#!/bin/bash
set -x
PROVISIONER_SERVER=$(facter -p provisioner_server)
MACADDRESS=$(/usr/bin/facter macaddress)
FQDN=$(facter fqdn)
PROVISIONER_HOST_STATE=maintenance

function update_provisioner_state {
  wget --tries=10 -O /dev/null -q \
  "http://${PROVISIONER_SERVER}/checkin/${MACADDRESS}/${FQDN}?state=${1// /+}"
}

# Function for the client to clean up its puppet cert before running puppet
clean_puppet_cert() {
  local url="https://${1}:8140/puppet-ca/v1/certificate_status/${2}"

  wget --method=PUT -O- -nv --no-hsts --tries=10 --no-proxy --no-check-certificate --header="Content-Type: text/pson" --body-data='{"desired_state":"revoked"}' "$url"
  wget --method=DELETE -O - -nv --no-hsts --tries=10 --no-proxy --no-check-certificate "$url"
  # when the puppet agent is invoked, it creates the new ssl information here
  rm -rf /etc/puppet/ssl
  rm -rf /etc/puppetlabs/puppet/ssl
  rm -rf /var/lib/puppet/ssl
}

# without that, easy_installed pip won't be found,
# and things will fail
export PATH=/usr/local/sbin:/usr/local/bin:/usr/sbin:/usr/bin:/sbin:/bin:/opt/puppetlabs/bin

# sleep until we see a gateway set.
while ! netstat -rn | grep ^0.0.0.0 ; do  echo no default gateway found ; sleep 15 ; done

# Let provisioner know that firstboot is starting
wget --method=PUT --no-hsts --tries=10 --no-proxy -O /dev/null -q --header='Content-Type: application/json' --body-data='{"username": "provisioner-state", "comment": "firstboot : starting : success"}' "https://${PROVISIONER_SERVER}/api/1.0/devices/${MACADDRESS}/comment"

for network_driver in $(facter -p network_drivers | tr ',' " ")
do
  [ -f "/opt/twitch/provision-files/${network_driver}.sh" ] && "/opt/twitch/provision-files/${network_driver}.sh"
done

# Get proxy vars if needed
if [ -f /etc/profile.d/proxy.sh ]; then
  # shellcheck disable=SC1091
  source /etc/profile.d/proxy.sh
fi

# install services first
apt-get install -y \
ladvd \
openssh-server

apt-get install -y \
apt-transport-https \
augeas-lenses \
augeas-tools \
build-essential \
curl \
git \
libuser \
libxml2-dev \
libaugeas-dev \
libaugeas0 \
man-db \
pkg-config \
python-setuptools \
software-properties-common \
vim \
lshw \
dmidecode \
freeipmi-tools

# fix /etc/hosts with hostname.domain
# shellcheck disable=SC2016
puppet apply -e 'host { "$hostname.": ensure => "absent", ip => "127.0.1.1", host_aliases => "$hostname" }'
# shellcheck disable=SC2016
puppet apply -e 'host { "$hostname.$domain": ensure => "present", ip => "127.0.1.1", host_aliases => "$hostname" }'

# check domain is set correctly
CHECK_DOMAIN=$(facter domain)

if [ -z "$CHECK_DOMAIN" ]; then
  update_provisioner_state "firstboot : error domain name empty : failure"
  exit 1
fi

# run lshw
echo 'Collecting hardware information...'
mkdir -p /etc/hardware
lshw -json 2>/dev/null > /etc/hardware/lshw.json

# checkin with mac address
update_provisioner_state 'firstboot : before puppet : success'

# determine currently configured puppet_ca server
# shellcheck disable=SC2153
puppet_ca_server="${PUPPET_CA_SERVER}"
# clean the old cert
if [ ! -z "$puppet_ca_server" ]; then
  echo "Cleaning certificate from ${puppet_ca_server}..."
  clean_puppet_cert "${puppet_ca_server}" "$(facter fqdn)"
fi

if [ "${RUN_PUPPET}" == 'true' ]
then
  puppet agent --enable
  update_provisioner_state 'firstboot : puppet run start : success'
  retry=0
  while [[ $retry -lt 3 ]]; do
    puppet agent --test \
      --environment="${PUPPET_ENVIRONMENT}" \
      --server="${PUPPET_SERVER}" \
      --ca_server="${PUPPET_CA_SERVER}"
    exitcode=$?
    retry=$(( retry + 1 ))
    ( [[ $exitcode -ne 0 ]] && sleep 5 ) || break
  done
  case $exitcode in
    0)
      msg="success: Puppet ran $retry times and the last run succeeded with no changes or failures; the system was already in the desired state"
      PROVISIONER_HOST_STATE=active
      ;;
    1)
      msg="failure: run failed or wasn't attempted due to another run already in progress"
      update_provisioner_state "firstboot : puppet finished run : ${msg}, exit: ${exitcode}"
      exit 1
      ;;
    2)
      msg="success: changes were made"
      PROVISIONER_HOST_STATE=active
      ;;
    [46])
      msg="partial failure: run completed, some resources failed"
      ;;
    *)
      msg="failure: puppet failed spectacularly"
      ;;
  esac
  update_provisioner_state "firstboot : puppet finished run : ${msg}, exit: ${exitcode}"
else
  update_provisioner_state 'firstboot : puppet not enabled : success'
fi

# recreate initramfs to make sure systemd links files are present
update-initramfs -k all -c

# checkin with mac address
update_provisioner_state 'firstboot : after puppet, imaging complete : success'

# run inventory-agent after puppet
systemctl start inventory-agent

rm /etc/cron.d/firstboot

# Let provisioner know firstboot is complete
wget --method=PUT --no-hsts --tries=10 --no-proxy -O /dev/null -q --header='Content-Type: application/json' --body-data='{"username": "provisioner-state", "comment": "firstboot : complete : success"}' "https://${PROVISIONER_SERVER}/api/1.0/devices/${MACADDRESS}/comment"
# Set provisioner host state
wget --method=PUT --no-hsts --tries=10 --no-proxy -O /dev/null -q --header='Content-Type: application/json' --body-data="{ 'identifier': ${MACADDRESS}, 'state': ${PROVISIONER_HOST_STATE} }" "https://${PROVISIONER_SERVER}/api/1.0/parameters/"
