#!/bin/bash

set -uex

# shellcheck disable=SC2012
query_name=$(ls frozen_*.tgz|sed -e 's/.tgz$//' -e 's/:/_/g')
REGION=us-west-2
WAIT_TIME_SECONDS=600

SECONDS=$(date +%s)

QUEUE_CLEANUP_TIMEOUT=86400 # 1 day

QUEUE_MAX_MESSAGE_SIZE=2048
QUEUE_MESSAGE_RETENTION_PERIOD=600
QUEUE_VISIBILITY_TIMEOUT=30

QUEUE_ATTRS="MaximumMessageSize=$QUEUE_MAX_MESSAGE_SIZE,\
    MessageRetentionPeriod=$QUEUE_MESSAGE_RETENTION_PERIOD,\
    VisibilityTimeout=$QUEUE_VISIBILITY_TIMEOUT"

case "$BRANCH_NAME" in
    main|master)
        env_suffix="prod"
        aws_account_ids=277437172290
        ;;
    *)
        env_suffix="staging"
        aws_account_ids=152548578290
        ;;
esac

queue_url="$(aws sqs create-queue \
    --region $REGION \
    --queue-name "$query_name" \
    --attributes "$QUEUE_ATTRS" \
    | jq -r .QueueUrl)"

aws --region $REGION sqs add-permission --queue-url "$queue_url" --aws-account-ids "$aws_account_ids" --actions SendMessage --label SendMessage

aws s3 cp frozen_*.tgz s3://puppet-environment-renderer-bucket-$env_suffix/

deadline=$((SECONDS + WAIT_TIME_SECONDS))

while ((SECONDS < deadline)); do
    message_json="$(aws sqs receive-message \
        --region "$REGION" \
        --queue-url "$queue_url" \
        --wait-time-seconds 10)"
    [ -n "$message_json" ] && break
    sleep 10
done

for url in $(aws --region $REGION sqs list-queues|jq -r .QueueUrls[]); do
    [[ $url =~ frozen_ ]] || continue
    # queue name contains the timestamp, but only alphanumerics, dashes and underscores allowed
    # so we need to reconstruct time to get seconds since epoch
    # $url should be in the form https://us-west-2.queue.amazonaws.com/181769184623/frozen_production_2021-09-21_20_44_35
    timestamp=$(date -d "$(basename "$url"|sed -e 's/.*_\(....\)-\(..\)-\(..\)_\(..\)_\(..\)_\(..\)/\1-\2-\3 \4:\5:\6/')" +%s)
    ((timestamp + QUEUE_CLEANUP_TIMEOUT < SECONDS)) && {
        aws sqs delete-queue --queue-url "$url" --region $REGION || true
    }
done
aws sqs delete-queue --queue-url "$queue_url" --region $REGION || true

# shellcheck disable=SC2162
while read line; do
    declare "$line"
done < <(echo "$message_json"|python -c 'import sys, json; print(json.load(sys.stdin)["Messages"][0]["Body"])')

[ "${status:-}" == "ok" ] || exit 1
exit 0
