package client

import (
	"CoralGoClient/dialer"
	"CoralGoCodec/codec"
	"github.com/pkg/errors"
)

type Client interface {
	Call(asmName, operation string, input interface{}, output interface{}, options ...RequestOption) error
}

type OperationRef struct {
	AssemblyName, Name string
}

type ServiceRef struct {
	AssemblyName, ServiceName string
}

type client struct {
	serviceRef codec.ShapeRef
	dialer     dialer.Dialer
	codec      codec.RoundTripper
	connPool   ConnectionPool
}

func NewClient(serviceAssembly, serviceName string, dialer dialer.Dialer, c codec.RoundTripper, options ...Option) Client {
	cl := &client{
		serviceRef: codec.ShapeRef{AsmName: serviceAssembly, ShapeName: serviceName},
		dialer:     dialer,
		codec:      c,
		connPool: newDefaultConnectionPool(
			DefaultPoolSize,
			DefaultMaxTTL,
			DefaultJitter,
			dialer.Dial), // Connection factory
	}
	cl.applyOptions(options)
	return cl
}

func (c *client) applyOptions(options []Option) {
	for _, o := range options {
		o(c)
	}
}

func (c *client) Call(asmName, operation string, in interface{}, out interface{}, options ...RequestOption) error {
	conn, err := c.connPool.Consume()
	if err != nil {
		return err
	}
	req := &codec.Request{
		Service:   c.serviceRef,
		Operation: codec.ShapeRef{AsmName: asmName, ShapeName: operation},
		Input:     in,
		Output:    out,
	}
	for i, opt := range options {
		if err := opt(req); err != nil {
			return errors.Wrapf(err, "failed to apply request option %d", i)
		}
	}
	err = c.codec.RoundTrip(req, conn)
	if err != nil {
		conn.Close()
		return err
	}
	c.connPool.Return(conn)
	return nil
}
