/* Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved. */

package env

import (
	"fmt"
	"os"
	"os/user"
	"path/filepath"
	"strings"

	"github.com/pkg/errors"

	"GoLog/log"
	"apollo"
)

const packageInfoFile = "packageInfo"

func GetRoot() (string, error) {
	// First attempt to pull the environment root from Apollo.
	root, err := getApolloRoot()
	if err != nil {
		log.Info(err)
	}
	if isValidRoot(root) {
		log.Debug("Using Apollo environment")
		return root, nil
	}

	// Next try to pull the environment root for a desktop.
	root, err = getWorkspaceRoot()
	if err != nil {
		log.Info(err)
	}
	if isValidRoot(root) {
		log.Debug("Using workspace environment")
		return root, nil
	}

	// Finally try to pull the environment root for a Regional ToD test run.
	absPath, err := filepath.Abs(".")
	if err != nil {
		return "", errors.Wrap(err, "get AAA root")
	}

	root, err = getRegionalToDWorkspaceRoot(absPath)
	if err != nil {
		return "", errors.Wrap(err, "get AAA root")
	}
	if isValidRoot(root) {
		log.Debug("Using ToD environment")
		return root, nil

	}

	// If reach here, we can't determine the AAA environment root
	return "", errors.New("get AAA root: cannot determine an environment root")
}

func getApolloRoot() (string, error) {
	env, err := apollo.CurEnvironment().PrimaryEnvironment()
	if err != nil {
		return "", fmt.Errorf("apollo root: %v", err)
	}

	// If we have an environment, then we need to be able to discover the root.
	root, err := env.Root()
	if err != nil {
		return "", fmt.Errorf("apollo root: %v", err)
	}

	return root, nil
}

// getWorkspaceRoot will walk up from the executable's directory until it finds a workspace's
// packageInfo file. It will then use this to drill into the AAA environment.
// Expected path format: <workspace>/AAA/<package>-<user>/
func getWorkspaceRoot() (string, error) {
	ex, err := os.Executable()
	if err != nil {
		return "", fmt.Errorf("workspace root: %v", err)
	}

	workspaceDir := filepath.Dir(ex)
	pastDirs := make([]string, 0)
	for !isFilePresent(filepath.Join(workspaceDir, packageInfoFile)) {
		parentDir, currentDir := filepath.Split(workspaceDir)

		// We're at the root of the file system
		if parentDir == "/" && currentDir == "" {
			return "", errors.New("workspace root: could not find a workspace")
		}
		pastDirs = append(pastDirs, currentDir)
		workspaceDir = filepath.Clean(parentDir)
	}

	// package name is second to last directory visited
	packageName := pastDirs[len(pastDirs)-2]

	userName, err := user.Current()
	if err != nil {
		return "", fmt.Errorf("workspace root: %v", err)
	}

	aaaDirectoryName := fmt.Sprintf("%s-%s", packageName, userName.Username)
	return filepath.Join(workspaceDir, "AAA", aaaDirectoryName), nil
}

// Brazil command line tools are not available in the Prod fabric so we are forced to parse the path for the information we need.
// The USER environment variable is not set because Regional ToD invokes a test run with "env -i", but it can be pulled from the file path.
// The file path we are parsing is in the format: /*/TodWorker-<User>/*/src/<Package Name>/*
// The path we are generating is: /*/TodWorker-<User>/*/AAA/<Package Name>-<User>
func getRegionalToDWorkspaceRoot(fp string) (string, error) {
	sep := string(filepath.Separator)
	pathElem := strings.Split(fp, sep)

	var userName string
	var wsRoot string
	var packageName string
	hasTod := false

	for i, elem := range pathElem {
		if !hasTod {
			if strings.HasPrefix(elem, "TodWorker-") {
				todUserElem := strings.Split(pathElem[i], "-")
				if len(todUserElem) != 2 {
					break
				}
				userName = todUserElem[1]
				hasTod = true
			}
		} else if elem == "src" {
			if i == len(pathElem)-1 {
				break
			}
			// First element is an empty string
			wsRoot = sep + strings.Join(pathElem[1:i], sep)
			packageName = pathElem[i+1]
			break
		}
	}

	if userName == "" || wsRoot == "" || packageName == "" {
		return "", errors.New("regional ToD root: path is not in the correct format: " + fp)
	}

	root := wsRoot + "/AAA/" + packageName + "-" + userName
	return root, nil
}

// isValidRoot will check for known bad roots
func isValidRoot(path string) bool {
	// Integration tests run from the SDETools or the BrazilCLI Apollo environment
	// depending on the version of the Brazil CLI being used.
	// The SD will not be located there so we skip this step and check the fallbacks.
	if path == "/apollo/env/SDETools/Apollo/Primary" || path == "/apollo/env/BrazilCLI/Apollo/Primary" {
		return false
	}

	// If there is neither a AAA nor a AAATLS directory, we don't have a valid root
	aaaPath := filepath.Join(path, "var", "state", "AAA")
	aaaTlsPath := filepath.Join(path, "var", "state", "aaatls")
	if !(isFilePresent(aaaPath) || isFilePresent(aaaTlsPath)) {
		return false
	}

	return true
}

func isFilePresent(path string) bool {
	if _, err := os.Stat(path); !os.IsNotExist(err) {
		return true
	}

	return false
}
