/* Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved. */

package env

import (
	"errors"
	"testing"
)

const (
	// correct paths
	shortPath          = "/local/TodWorker-user/root/src/packageName"
	longPath           = "/local/something/TodWorker-user/testruns/testrun_id/src/packageName/path/to/executable"
	twoSrcPath         = "/local/something/src/TodWorker-user/something/src/packageName"
	twoSrcAfterTodPath = "/local/TodWorker-vssadmin/testruns/4e23ad40/src/packageName/src/golang.a2z.com/pac/external/vis"

	// incorrect paths
	noSrcPath          = "/this/TodWorker-user/path/has/no/source/folder" // i.e. no src folder
	noUserPath         = "/local/src/this/filepath/has/no/user"
	srcBeforeTodFolder = "/local/src/TodWorker-User"

	wsRootForShortPath          = "/local/TodWorker-user/root/AAA/packageName-user"
	wsRootForLongPath           = "/local/something/TodWorker-user/testruns/testrun_id/AAA/packageName-user"
	wsRootForTwoSrcPath         = "/local/something/src/TodWorker-user/something/AAA/packageName-user"
	wsRootForTwoSrcAfterTodPath = "/local/TodWorker-vssadmin/testruns/4e23ad40/AAA/packageName-vssadmin"
)

var (
	errMsgBase            = "regional ToD root: path is not in the correct format: "
	errNoSrc              = errors.New(errMsgBase + noSrcPath)
	errNoUser             = errors.New(errMsgBase + noUserPath)
	errSrcBeforeTodFolder = errors.New(errMsgBase + srcBeforeTodFolder)
)

func TestGetRegionalToDWorkspaceRoot(t *testing.T) {
	tests := []struct {
		name        string
		path        string
		expected    string
		expectedErr error
	}{
		{name: "success: short path", path: shortPath, expected: wsRootForShortPath},
		{name: "success: long path", path: longPath, expected: wsRootForLongPath},
		{name: "success: `src` before and after tod folder", path: twoSrcPath, expected: wsRootForTwoSrcPath},
		{name: "success: 2 `src` after tod folder", path: twoSrcAfterTodPath, expected: wsRootForTwoSrcAfterTodPath},

		{name: "no scr folder", path: noSrcPath, expectedErr: errNoSrc},
		{name: "no username", path: noUserPath, expectedErr: errNoUser},
		{name: "'src' only before tod folder", path: srcBeforeTodFolder, expectedErr: errSrcBeforeTodFolder},
	}

	for _, tst := range tests {
		test := tst
		t.Run(test.name, func(t *testing.T) {
			t.Parallel()

			wsRoot, err := getRegionalToDWorkspaceRoot(test.path)

			if err != nil && test.expected != "" {
				t.Errorf("Expected wsRoot = '%v', but found err: '%v'", test.expected, err)
			}
			if wsRoot != "" && test.expectedErr != nil {
				t.Errorf("Expected err = '%v', but found wsRoot: '%v'", test.expectedErr, wsRoot)
			}
			if test.expectedErr != nil && err != nil && test.expectedErr.Error() != err.Error() {
				t.Errorf("Expected err = '%v', but got err = '%v'", test.expectedErr, err)
			}
			if test.expected != wsRoot {
				t.Errorf("Expected wsRoot = '%v', but got wsRoot = '%v'", test.expected, wsRoot)
			}
		})
	}

}
