package sd

import (
	"io/ioutil"
	"net"
	"path/filepath"
	"sync"
	"time"

	"github.com/pkg/errors"

	"GoLog/log"
	"aaa/internal/env"
)

const (
	dialTimeout = 1 * time.Second
	maxRetries  = 2
)

type Client struct {
	sync.Mutex
	socketPath string
	pool       []net.Conn
}

// Connect returns a socket connection with the Security Daemon.  A connection is
// returned from the connection pool if available, otherwise a new connection is created.
func (sd *Client) Connect() (net.Conn, error) {
	sd.Lock()
	defer sd.Unlock()
	numPooled := len(sd.pool)
	log.Trace("Number of pooled connections", numPooled)
	if numPooled > 0 {
		conn := sd.pool[0]
		copy(sd.pool, sd.pool[1:])
		sd.pool = sd.pool[:numPooled-1]
		log.Trace("Using pooled connection", conn)
		return conn, nil
	}
	c, err := net.DialTimeout("unix", sd.socketPath, dialTimeout)
	if err != nil {
		return nil, errors.Errorf("Failed to establish connection to AAASecurityDaemon with path %v: %v", sd.socketPath, err)
	}
	log.Trace("Created new connection", c)
	return c, nil
}

// Release puts the given connection back into the pool if it is not null.
// Connections that have caused an error should not be released.
func (sd *Client) Release(conn net.Conn) {
	if conn == nil {
		return
	}
	sd.Lock()
	defer sd.Unlock()
	log.Trace("Returning connection to the pool", conn)
	sd.pool = append(sd.pool, conn)
}

func NewSDClient(options ...func(*Client) error) (*Client, error) {
	sd := &Client{pool: make([]net.Conn, 0, 5)}
	for _, option := range options {
		err := option(sd)
		if err != nil {
			log.Errorf("Error creating AAASecurityDaemon client. Error while setting options: %+v", err)
			return nil, errors.Errorf("Error creating AAASecurityDaemon client. Error while setting options: %v", err)
		}
	}
	if sd.socketPath == "" {
		log.Error("Error creating AAASecurityDaemon client. The socketPath must be specified.")
		return nil, errors.Errorf("Error creating AAASecurityDaemon client. The socketPath must be specified.")
	}
	return sd, nil
}

// StaticSocketPath is a constructor option that allows you to specify what path to use
// for the socket.
func StaticSocketPath(path string) func(*Client) error {
	return func(sd *Client) error {
		sd.socketPath = path
		return nil
	}
}

// SocketPathAutoDiscovery is a constructor option that tells the client to attempt to determine the
// socket path to use from the current Apollo environment.  If no environment is found, then an error
// is returned which causes initialization of the client to fail.
func SocketPathAutodiscovery() func(*Client) error {
	return SocketPathDiscovery(filepath.Join("var", "state", "AAA", "be.sock.txt"))
}

// SocketPathDiscovery is a constructor option that tells the client to attempt to determine the socket path to use
// from the root of the current Apollo environment and the path suffix passed in. If no environment is found, then an
// error is returned which causes initialization of the client to fail.
func SocketPathDiscovery(path string) func(*Client) error {
	return func(sd *Client) error {
		root, err := env.GetRoot()
		if err != nil {
			return errors.Errorf("Unable to determine root: %v", err)
		}
		log.Debug("Using root", root)
		beSockPath := filepath.Join(root, path)
		log.Debug("Reading socket file at", beSockPath)
		buff, err := ioutil.ReadFile(beSockPath)
		if err != nil {
			errors.Errorf("Unable to read the be.sock.txt file located at %v: %v", beSockPath, err)
		}
		sd.socketPath = string(buff)
		log.Debug("Set socketPath to", sd.socketPath)
		return nil
	}
}

func (sd *Client) SanityCheck() error {
	c, err := sd.Connect()
	if c != nil {
		c.Close()
	}
	return err
}
