/* Copyright 2016 Amazon.com, Inc. or its affiliates. All Rights Reserved. */
package sd

import (
	"aaa"
	"bytes"
	"io"
	"io/ioutil"
	"math/rand"
	"net"
	"net/http"
	"os"
	"path/filepath"
	"strconv"
	"testing"
	"time"

	"github.com/pkg/errors"
)

const (
	testOperation = "TestOperation"
	testService   = "TestService"
)

func TestNewErrorCases(t *testing.T) {
	// No options
	empty := []func(*Client) error{}
	if _, err := NewSDClient(empty...); err == nil {
		t.Error("Expected an error for no options")
	}
	// Auto-Discovery will fail because we're not in an Apollo env
	if _, err := NewSDClient(SocketPathAutodiscovery()); err == nil {
		t.Error("Expected an error for constructing outside of an Apollo Environment")
	}
}

func TestAuthorizeRequest(t *testing.T) {
	path, server := createServer(t)
	defer os.Remove(path)
	defer server.Close()

	sd, err := NewSDClient(StaticSocketPath(path))
	if err != nil {
		t.Fatal("Unable to create SD client", err)
	}

	ts := &testServer{}
	go ts.listen(server, t)

	// Verify that our connection is good
	if err := sd.SanityCheck(); err != nil {
		t.Fatal("Expected no error performing the SanityCheck, instead found", err)
	}

	sctx := &aaa.ServiceContext{
		Service:   testService,
		Operation: testOperation,
	}

	tests := []struct {
		name       string
		body       io.ReadCloser
		serverData []byte
		serverErr  error
		retainConn bool
	}{
		{"Valid request", ioutil.NopCloser(bytes.NewBuffer([]byte(testOperation))), testValidAuthorizeRequestData, nil, true},
		{"Valid request 2", ioutil.NopCloser(bytes.NewBuffer([]byte(testOperation))), testValidAuthorizeRequestData, nil, true},
		{"Invalid data", ioutil.NopCloser(bytes.NewBuffer([]byte(testOperation))), testInvalidData, nil, false},
		{"Valid request 3", ioutil.NopCloser(bytes.NewBuffer([]byte(testOperation))), testValidAuthorizeRequestData, nil, true},
		{"Invalid data 2", ioutil.NopCloser(bytes.NewBuffer([]byte(testOperation))), testInvalidData, nil, false},
		{"Valid request 4", ioutil.NopCloser(bytes.NewBuffer([]byte(testOperation))), testValidAuthorizeRequestData, nil, true},
	}

	expectedConns := 0
	for _, test := range tests {
		test := test
		t.Run(test.name, func(t *testing.T) {
			if test.retainConn {
				expectedConns = 1
			} else {
				expectedConns = 0
			}
			ts.set(test.serverData, test.serverErr)
			result, err := sd.AuthorizeRequest(sctx)
			if test.retainConn != (err == nil) {
				t.Error(test.name, "- Unexpected error state:", err)
			}
			if test.retainConn != (result != nil) {
				t.Error(test.name, "- Unexpected result state:", result)
			}
			if len(sd.pool) != expectedConns {
				t.Error(test.name, "- Expected there to be", expectedConns, "pooled connections but found", len(sd.pool))
			}
		})
	}
}

func TestEncodeRequest(t *testing.T) {
	path, server := createServer(t)
	defer os.Remove(path)
	defer server.Close()

	sd, err := NewSDClient(StaticSocketPath(path))
	if err != nil {
		t.Fatal("Unable to create SD client", err)
	}

	ts := &testServer{}
	go ts.listen(server, t)

	// Verify that our connection is good
	if err := sd.SanityCheck(); err != nil {
		t.Fatal("Expected no error performing the SanityCheck, instead found", err)
	}

	tests := []struct {
		name       string
		body       io.ReadCloser
		serverData []byte
		serverErr  error
		retainConn bool
	}{
		{"Bad body", &erroringReader{}, nil, nil, false},
		{"Valid request", ioutil.NopCloser(bytes.NewBuffer([]byte(testOperation))), testValidEncodeRequestData, nil, true},
		{"Valid request 2", ioutil.NopCloser(bytes.NewBuffer([]byte(testOperation))), testValidEncodeRequestData, nil, true},
		{"Invalid data", ioutil.NopCloser(bytes.NewBuffer([]byte(testOperation))), testInvalidData, nil, false},
		{"Valid request 3", ioutil.NopCloser(bytes.NewBuffer([]byte(testOperation))), testValidEncodeRequestData, nil, true},
		{"Invalid data 2", ioutil.NopCloser(bytes.NewBuffer([]byte(testOperation))), testInvalidData, nil, false},
		{"Valid request 4", ioutil.NopCloser(bytes.NewBuffer([]byte(testOperation))), testValidEncodeRequestData, nil, true},
	}

	expectedConns := 0
	for _, test := range tests {
		test := test
		t.Run(test.name, func(t *testing.T) {
			if test.retainConn {
				expectedConns = 1
			} else {
				expectedConns = 0
			}
			r, _ := http.NewRequest("GET", "http://example.com", nil)
			r.Body = test.body
			ts.set(test.serverData, test.serverErr)
			ctx, err := sd.EncodeRequest(testService, testOperation, r)
			if test.retainConn != (err == nil) {
				t.Error(test.name, "- Unexpected error state:", err)
			}
			if test.retainConn != (ctx != nil) {
				t.Error(test.name, "- Unexpected context state:", ctx)
			}
			if len(sd.pool) != expectedConns {
				t.Error(test.name, "- Expected there to be", expectedConns, "pooled connections but found", len(sd.pool))
			}
		})
	}
}

func TestEncodeResponse(t *testing.T) {
	path, server := createServer(t)
	defer os.Remove(path)
	defer server.Close()

	sd, err := NewSDClient(StaticSocketPath(path))
	if err != nil {
		t.Fatal("Unable to create SD client", err)
	}

	ts := &testServer{}
	go ts.listen(server, t)

	// Verify that our connection is good
	if err := sd.SanityCheck(); err != nil {
		t.Fatal("Expected no error performing the SanityCheck, instead found", err)
	}

	sctx := &aaa.ServiceContext{
		Service:   testService,
		Operation: testOperation,
	}

	tests := []struct {
		name       string
		body       []byte
		serverData []byte
		serverErr  error
		retainConn bool
	}{
		{"Valid request", []byte(testOperation), testValidEncodeResponseData, nil, true},
		{"Valid request 2", []byte(testOperation), testValidEncodeResponseData, nil, true},
		{"Invalid data", []byte(testOperation), testInvalidData, nil, false},
		{"Valid request 3", []byte(testOperation), testValidEncodeResponseData, nil, true},
		{"Invalid data 2", []byte(testOperation), testInvalidData, nil, false},
		{"Valid request 4", []byte(testOperation), testValidEncodeResponseData, nil, true},
	}

	expectedConns := 0
	for _, test := range tests {
		test := test
		t.Run(test.name, func(t *testing.T) {
			if test.retainConn {
				expectedConns = 1
			} else {
				expectedConns = 0
			}
			ts.set(test.serverData, test.serverErr)
			data, err := sd.EncodeResponse(sctx, http.Header{}, test.body)
			if test.retainConn != (err == nil) {
				t.Error(test.name, "- Unexpected error state:", err)
			}
			if test.retainConn != (len(data) != 0) {
				t.Error(test.name, "- Unexpected data state:", data)
			}
			if len(sd.pool) != expectedConns {
				t.Error(test.name, "- Expected there to be", expectedConns, "pooled connections but found", len(sd.pool))
			}
		})
	}
}

func TestDecodeRequest(t *testing.T) {
	path, server := createServer(t)
	defer os.Remove(path)
	defer server.Close()

	sd, err := NewSDClient(StaticSocketPath(path))
	if err != nil {
		t.Fatal("Unable to create SD client", err)
	}

	ts := &testServer{}
	go ts.listen(server, t)

	// Verify that our connection is good
	if err := sd.SanityCheck(); err != nil {
		t.Fatal("Expected no error performing the SanityCheck, instead found", err)
	}

	tests := []struct {
		name       string
		body       io.ReadCloser
		serverData []byte
		serverErr  error
		retainConn bool
	}{
		{"Bad body", &erroringReader{}, nil, nil, false},
		{"Valid request", ioutil.NopCloser(bytes.NewBuffer([]byte(testOperation))), testValidDecodeRequestData, nil, true},
		{"Valid request 2", ioutil.NopCloser(bytes.NewBuffer([]byte(testOperation))), testValidDecodeRequestData, nil, true},
		{"Invalid data", ioutil.NopCloser(bytes.NewBuffer([]byte(testOperation))), testInvalidData, nil, false},
		{"Valid request 3", ioutil.NopCloser(bytes.NewBuffer([]byte(testOperation))), testValidDecodeRequestData, nil, true},
		{"Invalid data 2", ioutil.NopCloser(bytes.NewBuffer([]byte(testOperation))), testInvalidData, nil, false},
		{"Valid request 4", ioutil.NopCloser(bytes.NewBuffer([]byte(testOperation))), testValidDecodeRequestData, nil, true},
	}

	expectedConns := 0
	for _, test := range tests {
		test := test
		t.Run(test.name, func(t *testing.T) {
			if test.retainConn {
				expectedConns = 1
			} else {
				expectedConns = 0
			}
			r, _ := http.NewRequest("GET", "http://example.com", test.body)
			ts.set(test.serverData, test.serverErr)
			ctx, err := sd.DecodeRequest(r)
			t.Log(ctx)
			if test.retainConn != (err == nil) {
				t.Error(test.name, "- Unexpected error state:", err)
			}
			if len(sd.pool) != expectedConns {
				t.Error(test.name, "- Expected there to be", expectedConns, "pooled connections but found", len(sd.pool))
			}
		})
	}
}

func TestDecodeResponse(t *testing.T) {
	path, server := createServer(t)
	defer os.Remove(path)
	defer server.Close()

	sd, err := NewSDClient(StaticSocketPath(path))
	if err != nil {
		t.Fatal("Unable to create SD client", err)
	}

	ts := &testServer{}
	go ts.listen(server, t)

	// Verify that our connection is good
	if err := sd.SanityCheck(); err != nil {
		t.Fatal("Expected no error performing the SanityCheck, instead found", err)
	}

	clientCxt := &aaa.ClientContext{
		KeyId:      "keyId",
		KeyVersion: 1,
		Service:    testService,
		Operation:  testOperation,
	}

	tests := []struct {
		name       string
		body       io.ReadCloser
		serverData []byte
		serverErr  error
		retainConn bool
	}{
		{"Bad body", &erroringReader{}, nil, nil, false},
		{"Valid request", ioutil.NopCloser(bytes.NewBuffer([]byte(testOperation))), testValidDecodeResponseData, nil, true},
		{"Valid request 2", ioutil.NopCloser(bytes.NewBuffer([]byte(testOperation))), testValidDecodeResponseData, nil, true},
		{"Invalid data", ioutil.NopCloser(bytes.NewBuffer([]byte(testOperation))), testInvalidData, nil, false},
		{"Valid request 3", ioutil.NopCloser(bytes.NewBuffer([]byte(testOperation))), testValidDecodeResponseData, nil, true},
		{"Invalid data 2", ioutil.NopCloser(bytes.NewBuffer([]byte(testOperation))), testInvalidData, nil, false},
		{"Valid request 4", ioutil.NopCloser(bytes.NewBuffer([]byte(testOperation))), testValidDecodeResponseData, nil, true},
	}

	expectedConns := 0
	for _, test := range tests {
		test := test
		t.Run(test.name, func(t *testing.T) {
			if test.retainConn {
				expectedConns = 1
			} else {
				expectedConns = 0
			}
			r := &http.Response{Body: test.body}
			ts.set(test.serverData, test.serverErr)
			err := sd.DecodeResponse(clientCxt, r)
			if test.retainConn != (err == nil) {
				t.Error(test.name, "- Unexpected error state:", err)
			}
			if len(sd.pool) != expectedConns {
				t.Error(test.name, "- Expected there to be", expectedConns, "pooled connections but found", len(sd.pool))
			}
		})
	}
}

func createServer(t *testing.T) (string, *net.UnixListener) {
	rand.Seed(time.Now().Unix())
	suffix := rand.Int31()
	path := filepath.Join(os.TempDir(), "sd_test"+strconv.Itoa(int(suffix))+".sock")
	t.Log("Using socket path", path)
	server, err := net.ListenUnix("unix", &net.UnixAddr{Name: path, Net: "unix"})
	if err != nil {
		os.Remove(path)
		t.Fatal(err)
	}
	return path, server
}

// testServer allows us to start up a UnixListener that can accept multiple simultaneous
// connections while customizing what the output of each connection will be.
type testServer struct {
	data []byte
	err  error
}

func (ts *testServer) set(data []byte, err error) {
	ts.data = data
	ts.err = err
}

func (ts *testServer) listen(server *net.UnixListener, t *testing.T) {
	for {
		conn, err := server.AcceptUnix()
		if err == nil {
			go ts.serve(conn, t)
		} else {
			t.Log("Error accepting:", err)
			break
		}
	}
}

func (ts *testServer) serve(c net.Conn, t *testing.T) {
	defer c.Close()
	buf := make([]byte, 2048)
	for {
		if ts.err != nil {
			t.Log("ts.err is not nil, closing the connection")
			return
		} else if len(ts.data) > 0 {
			c.SetDeadline(time.Now().Add(250 * time.Millisecond))
			n, err := c.Read(buf)
			if n == 0 || (err != nil && err != io.EOF) {
				continue
			}
			t.Log("testServer read", n, "bytes")
			n, err = c.Write(ts.data[:16])
			t.Log("testServer wrote", n, "bytes")
			if err != nil {
				t.Log("writing failed, closing the connection", err)
				continue
			}
			if len(ts.data) > 24 {
				n, err = c.Write(ts.data[16:])
				t.Log("testServer wrote", n, "bytes")
				if err != nil {
					t.Log("writing failed, closing the connection", err)
					continue
				}
			}
		}
	}
}

type erroringReader struct {
}

func (*erroringReader) Read([]byte) (int, error) {
	return 0, errors.New("erroringReader")
}

func (*erroringReader) Close() error {
	return nil
}

var (
	testInvalidData               = []byte{66, 69, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}
	testValidAuthorizeRequestData = []byte{
		// Header
		0x42, 0x45, 0x0, 0x0, 0x0, 0x0, 0x0, 0x9e, 0x1, 0x0, 0x1, 0x0, 0x0, 0x0, 0x0, 0x6,
		// Service layer
		0x92, 0x90, 0x92, 0x1, 0xcc, 0x98, 0x93,
		// Data layer
		0x90, 0x92, 0x1, 0x0, 0x92, 0x2, 0xda, 0x0, 0x8e, 0x41, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x69, 0x7a, 0x65, 0x64, 0x3a,
		0x20, 0x52, 0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 0x20, 0x66, 0x72, 0x6f, 0x6d, 0x20, 0x61, 0x6d, 0x7a, 0x6e, 0x31,
		0x2e, 0x61, 0x61, 0x61, 0x2e, 0x69, 0x64, 0x2e, 0x78, 0x63, 0x6b, 0x74, 0x6f, 0x35, 0x36, 0x73, 0x77, 0x79, 0x6e,
		0x66, 0x68, 0x65, 0x75, 0x71, 0x6c, 0x6b, 0x72, 0x66, 0x70, 0x74, 0x6e, 0x6c, 0x6e, 0x71, 0x20, 0x66, 0x6f, 0x72,
		0x20, 0x56, 0x43, 0x50, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x73, 0x53, 0x65, 0x72, 0x76, 0x69,
		0x63, 0x65, 0x2e, 0x69, 0x73, 0x41, 0x70, 0x70, 0x72, 0x6f, 0x76, 0x61, 0x6c, 0x4d, 0x61, 0x6e, 0x61, 0x67, 0x65,
		0x72, 0x49, 0x64, 0x56, 0x61, 0x6c, 0x69, 0x64, 0x20, 0x69, 0x73, 0x20, 0x73, 0x75, 0x63, 0x63, 0x65, 0x73, 0x73,
		0x66, 0x75, 0x6c, 0x6c, 0x79, 0x20, 0x61, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x69, 0x7a, 0x65, 0x64, 0x2e,
	}
	testValidDecodeRequestData = []byte{
		// Header
		0x42, 0x45, 0x0, 0x0, 0x0, 0x0, 0x1, 0xeb, 0x1, 0x0, 0x1, 0x0, 0x0, 0x0, 0x0, 0x7,
		// Service layer
		0x92, 0x90, 0x92, 0x1, 0xcd, 0x1, 0xe4,
		// Data layer
		0x98, 0x90, 0x92, 0x1, 0xda, 0x0, 0xbd, 0x7b, 0x22, 0x4f, 0x70, 0x65, 0x72, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x22, 0x3a,
		0x22, 0x63, 0x6f, 0x6d, 0x2e, 0x61, 0x6d, 0x61, 0x7a, 0x6f, 0x6e, 0x2e, 0x76, 0x63, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x73,
		0x73, 0x69, 0x6f, 0x6e, 0x73, 0x73, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x23, 0x69, 0x73, 0x41, 0x70, 0x70, 0x72, 0x6f,
		0x76, 0x61, 0x6c, 0x4d, 0x61, 0x6e, 0x61, 0x67, 0x65, 0x72, 0x49, 0x64, 0x56, 0x61, 0x6c, 0x69, 0x64, 0x22, 0x2c, 0x22,
		0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x22, 0x3a, 0x22, 0x63, 0x6f, 0x6d, 0x2e, 0x61, 0x6d, 0x61, 0x7a, 0x6f, 0x6e,
		0x2e, 0x76, 0x63, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x73, 0x73, 0x65, 0x72, 0x76, 0x69, 0x63,
		0x65, 0x23, 0x56, 0x43, 0x50, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x73, 0x53, 0x65, 0x72, 0x76, 0x69,
		0x63, 0x65, 0x22, 0x2c, 0x22, 0x49, 0x6e, 0x70, 0x75, 0x74, 0x22, 0x3a, 0x7b, 0x22, 0x75, 0x73, 0x65, 0x72, 0x49, 0x64,
		0x22, 0x3a, 0x22, 0x6d, 0x72, 0x75, 0x74, 0x74, 0x65, 0x72, 0x22, 0x2c, 0x22, 0x6d, 0x61, 0x6e, 0x61, 0x67, 0x65, 0x72,
		0x49, 0x64, 0x22, 0x3a, 0x22, 0x77, 0x61, 0x6e, 0x67, 0x6c, 0x75, 0x63, 0x79, 0x22, 0x7d, 0x7d, 0x92, 0x2, 0xda, 0x0,
		0x27, 0x61, 0x6d, 0x7a, 0x6e, 0x31, 0x2e, 0x61, 0x61, 0x61, 0x2e, 0x69, 0x64, 0x2e, 0x78, 0x63, 0x6b, 0x74, 0x6f, 0x35,
		0x36, 0x73, 0x77, 0x79, 0x6e, 0x66, 0x68, 0x65, 0x75, 0x71, 0x6c, 0x6b, 0x72, 0x66, 0x70, 0x74, 0x6e, 0x6c, 0x6e, 0x71,
		0x92, 0x3, 0xda, 0x0, 0x2a, 0x41, 0x70, 0x6f, 0x6c, 0x6c, 0x6f, 0x45, 0x6e, 0x76, 0x3a, 0x43, 0x6f, 0x72, 0x61, 0x6c,
		0x44, 0x69, 0x76, 0x65, 0x72, 0x50, 0x72, 0x6f, 0x78, 0x79, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x2f, 0x43, 0x4f,
		0x52, 0x50, 0x2f, 0x50, 0x72, 0x6f, 0x64, 0x92, 0x4, 0xb1, 0x43, 0x6f, 0x72, 0x61, 0x6c, 0x44, 0x69, 0x76, 0x65, 0x72,
		0x57, 0x65, 0x62, 0x73, 0x69, 0x74, 0x65, 0x92, 0x5, 0x93, 0x90, 0x92, 0x1, 0xc3, 0x92, 0x2, 0xc3, 0x92, 0x6, 0x93, 0x90,
		0x92, 0x1, 0xb4, 0x56, 0x43, 0x50, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x73, 0x53, 0x65, 0x72, 0x76,
		0x69, 0x63, 0x65, 0x92, 0x2, 0xb8, 0x69, 0x73, 0x41, 0x70, 0x70, 0x72, 0x6f, 0x76, 0x61, 0x6c, 0x4d, 0x61, 0x6e, 0x61,
		0x67, 0x65, 0x72, 0x49, 0x64, 0x56, 0x61, 0x6c, 0x69, 0x64, 0x92, 0x7, 0x93, 0x90, 0x92, 0x1, 0x1, 0x92, 0x2, 0xda, 0x0,
		0x65, 0x63, 0x6f, 0x6d, 0x2e, 0x61, 0x6d, 0x61, 0x7a, 0x6f, 0x6e, 0x2e, 0x61, 0x61, 0x61, 0x2e, 0x56, 0x43, 0x50, 0x65,
		0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x73, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x2e, 0x43, 0x6f, 0x72,
		0x61, 0x6c, 0x44, 0x69, 0x76, 0x65, 0x72, 0x57, 0x65, 0x62, 0x73, 0x69, 0x74, 0x65, 0x2e, 0x61, 0x6d, 0x7a, 0x6e, 0x31,
		0x2e, 0x61, 0x61, 0x61, 0x2e, 0x69, 0x64, 0x2e, 0x78, 0x63, 0x6b, 0x74, 0x6f, 0x35, 0x36, 0x73, 0x77, 0x79, 0x6e, 0x66,
		0x68, 0x65, 0x75, 0x71, 0x6c, 0x6b, 0x72, 0x66, 0x70, 0x74, 0x6e, 0x6c, 0x6e, 0x71, 0x2e, 0x44, 0x65, 0x66, 0x61, 0x75,
		0x6c, 0x74,
	}
	testValidDecodeResponseData = []byte{
		// Header
		66, 69, 0, 0, 0, 0, 3, 11, 1, 0, 1, 0, 0, 0, 0, 7,
		// Service layer
		146, 144, 146, 1, 205, 3, 5,
		// Data layer
		146, 144, 146, 1, 218, 2, 254, 0, 0, 2, 254, 153, 85, 238, 170, 8, 82, 101, 115, 117, 108, 116, 115, 0, 1, 122, 0, 0, 2,
		60, 153, 85, 238, 170, 15, 42, 118, 101, 99, 116, 111, 114, 108, 101, 110, 103, 116, 104, 42, 0, 11, 4, 0, 0, 0, 1, 4,
		118, 95, 48, 0, 1, 122, 0, 0, 2, 23, 153, 85, 238, 170, 10, 66, 83, 70, 68, 86, 97, 108, 105, 100, 0, 8, 7, 78, 111, 114,
		109, 97, 108, 0, 8, 42, 118, 97, 108, 117, 101, 42, 0, 1, 122, 0, 0, 1, 212, 153, 85, 238, 170, 14, 97, 99, 116, 105, 118,
		101, 83, 101, 115, 115, 105, 111, 110, 0, 9, 1, 0, 10, 118, 97, 108, 105, 100, 85, 66, 73, 68, 0, 8, 20, 56, 49, 48, 45, 51,
		50, 54, 52, 54, 55, 53, 45, 56, 49, 50, 52, 53, 53, 57, 0, 17, 42, 99, 108, 97, 115, 115, 72, 105, 101, 114, 97, 114, 99,
		104, 121, 42, 0, 1, 122, 0, 0, 0, 111, 153, 85, 238, 170, 15, 42, 118, 101, 99, 116, 111, 114, 108, 101, 110, 103, 116, 104,
		42, 0, 11, 4, 0, 0, 0, 2, 4, 118, 95, 48, 0, 8, 50, 105, 100, 101, 110, 116, 105, 116, 121, 46, 115, 101, 115, 115, 105, 111,
		110, 115, 101, 114, 118, 105, 99, 101, 118, 50, 46, 86, 97, 108, 105, 100, 97, 116, 101, 83, 101, 115, 115, 105, 111, 110,
		82, 101, 115, 112, 111, 110, 115, 101, 0, 4, 118, 95, 49, 0, 8, 17, 106, 97, 118, 97, 46, 108, 97, 110, 103, 46, 79, 98, 106,
		101, 99, 116, 0, 15, 118, 97, 108, 105, 100, 83, 101, 115, 115, 105, 111, 110, 73, 68, 0, 8, 20, 56, 49, 49, 45, 57, 55, 52,
		54, 48, 55, 48, 45, 53, 48, 50, 54, 49, 53, 50, 0, 8, 99, 111, 109, 109, 101, 110, 116, 0, 8, 79, 99, 111, 111, 107, 105, 101,
		83, 101, 115, 115, 105, 111, 110, 73, 100, 32, 105, 115, 32, 118, 97, 108, 105, 100, 59, 32, 105, 115, 67, 97, 99, 104, 101,
		97, 98, 108, 101, 83, 101, 115, 115, 105, 111, 110, 59, 32, 115, 101, 115, 115, 105, 111, 110, 39, 115, 32, 99, 117, 115, 116,
		111, 109, 101, 114, 32, 105, 100, 32, 105, 115, 32, 118, 97, 108, 105, 100, 59, 32, 0, 13, 99, 114, 101, 97, 116, 105, 111,
		110, 68, 97, 116, 101, 0, 13, 8, 65, 213, 219, 81, 54, 0, 0, 0, 16, 118, 97, 108, 105, 100, 67, 117, 115, 116, 111, 109, 101,
		114, 73, 68, 0, 8, 21, 65, 48, 55, 48, 57, 56, 55, 54, 75, 74, 87, 54, 53, 68, 56, 67, 78, 52, 85, 73, 0, 12, 42, 99, 108, 97,
		115, 115, 78, 97, 109, 101, 42, 0, 8, 50, 105, 100, 101, 110, 116, 105, 116, 121, 46, 115, 101, 115, 115, 105, 111, 110, 115,
		101, 114, 118, 105, 99, 101, 118, 50, 46, 86, 97, 108, 105, 100, 97, 116, 101, 83, 101, 115, 115, 105, 111, 110, 82, 101, 115,
		112, 111, 110, 115, 101, 0, 17, 99, 97, 99, 104, 101, 97, 98, 108, 101, 83, 101, 115, 115, 105, 111, 110, 0, 9, 1, 1, 17, 42,
		67, 111, 100, 105, 103, 111, 80, 114, 111, 116, 111, 99, 111, 108, 42, 0, 13, 8, 63, 241, 153, 153, 153, 153, 153, 154, 8, 83,
		101, 114, 118, 105, 99, 101, 0, 8, 17, 83, 101, 115, 115, 105, 111, 110, 83, 101, 114, 118, 105, 99, 101, 86, 50, 0, 7, 77,
		101, 116, 104, 111, 100, 0, 8, 16, 86, 97, 108, 105, 100, 97, 116, 101, 83, 101, 115, 115, 105, 111, 110, 0, 11, 73, 100, 101,
		110, 116, 105, 116, 105, 101, 115, 0, 1, 122, 0, 0, 0, 107, 153, 85, 238, 170, 12, 73, 109, 109, 101, 100, 67, 97, 108, 108,
		101, 114, 0, 1, 122, 0, 0, 0, 84, 153, 85, 238, 170, 10, 83, 117, 98, 106, 101, 99, 116, 73, 68, 0, 8, 47, 97, 109, 122, 110,
		49, 58, 115, 105, 100, 58, 48, 48, 48, 48, 48, 48, 48, 48, 45, 48, 48, 48, 48, 45, 48, 48, 48, 48, 45, 48, 48, 48, 48, 45, 48,
		48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 0, 5, 78, 97, 109, 101, 0, 8, 8, 117, 110, 107, 110, 111, 119, 110,
	}
	testValidEncodeResponseData = []byte{
		// Header
		0x42, 0x45, 0x0, 0x0, 0x0, 0x0, 0x1, 0xea, 0x1, 0x0, 0x1, 0x0, 0x0, 0x0, 0x0, 0x7,
		// Service layer
		0x92, 0x90, 0x92, 0x1, 0xcd, 0x1, 0xe3,
		// Data layer
		0x93, 0x90, 0x92, 0x1, 0xda, 0x0, 0x90, 0x7f, 0xf7, 0x48, 0x40, 0xbb, 0xd6, 0xd6, 0x8e, 0xfb, 0x6b, 0x52, 0xaa, 0x8a, 0xac, 0xe6,
		0x9c, 0x63, 0xf2, 0x46, 0x8f, 0xe0, 0x2e, 0x9c, 0x52, 0x2f, 0x99, 0xa6, 0xc1, 0x73, 0x6a, 0xae, 0xc2, 0x2d, 0xe7, 0xbb, 0x5d,
		0xc6, 0xa4, 0xa7, 0xb2, 0xab, 0xbf, 0x94, 0xa7, 0xe0, 0x69, 0xbd, 0x79, 0xa8, 0x49, 0x8d, 0x5d, 0x7c, 0xd7, 0x24, 0xd, 0x1c,
		0x41, 0xdb, 0x54, 0x7b, 0x31, 0x10, 0xeb, 0x9f, 0x6e, 0x50, 0x57, 0x61, 0x9a, 0xbd, 0x9d, 0xfb, 0xfc, 0xad, 0x1b, 0x53, 0x16,
		0xdd, 0x7d, 0x6c, 0xdc, 0x6, 0xcc, 0x5, 0x85, 0x59, 0xe8, 0xfc, 0xcb, 0x1, 0x52, 0x19, 0xbd, 0x9b, 0x92, 0xba, 0x82, 0xd3,
		0xc5, 0xd, 0x58, 0x93, 0xf1, 0x44, 0xb6, 0x7f, 0xe7, 0x71, 0xed, 0x9b, 0x71, 0x8a, 0x73, 0x2d, 0xb7, 0x3b, 0xac, 0xdb, 0xac,
		0xa2, 0x50, 0x60, 0x30, 0xe0, 0x2d, 0x19, 0xa, 0x8b, 0xca, 0xff, 0xc3, 0x66, 0x80, 0x48, 0xd5, 0xf6, 0x71, 0xc9, 0xda, 0xb,
		0x7c, 0xf5, 0x1c, 0x92, 0x2, 0xda, 0x1, 0x47, 0x41, 0x41, 0x41, 0x20, 0x69, 0x64, 0x65, 0x6e, 0x74, 0x69, 0x74, 0x79, 0x3d,
		0x63, 0x6f, 0x6d, 0x2e, 0x61, 0x6d, 0x61, 0x7a, 0x6f, 0x6e, 0x2e, 0x61, 0x61, 0x61, 0x2e, 0x56, 0x43, 0x50, 0x65, 0x72, 0x6d,
		0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x73, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x2e, 0x43, 0x6f, 0x72, 0x61, 0x6c, 0x44,
		0x69, 0x76, 0x65, 0x72, 0x57, 0x65, 0x62, 0x73, 0x69, 0x74, 0x65, 0x2e, 0x61, 0x6d, 0x7a, 0x6e, 0x31, 0x2e, 0x61, 0x61, 0x61,
		0x2e, 0x69, 0x64, 0x2e, 0x78, 0x63, 0x6b, 0x74, 0x6f, 0x35, 0x36, 0x73, 0x77, 0x79, 0x6e, 0x66, 0x68, 0x65, 0x75, 0x71, 0x6c,
		0x6b, 0x72, 0x66, 0x70, 0x74, 0x6e, 0x6c, 0x6e, 0x71, 0x2e, 0x44, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x2f, 0x31, 0x2c, 0x20,
		0x4f, 0x70, 0x65, 0x72, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x3d, 0x69, 0x73, 0x41, 0x70, 0x70, 0x72, 0x6f, 0x76, 0x61, 0x6c, 0x4d,
		0x61, 0x6e, 0x61, 0x67, 0x65, 0x72, 0x49, 0x64, 0x56, 0x61, 0x6c, 0x69, 0x64, 0x2c, 0x20, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63,
		0x65, 0x3d, 0x56, 0x43, 0x50, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x73, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63,
		0x65, 0x2c, 0x20, 0x45, 0x6e, 0x63, 0x72, 0x79, 0x70, 0x74, 0x65, 0x64, 0x3d, 0x74, 0x72, 0x75, 0x65, 0x2c, 0x20, 0x53, 0x69,
		0x67, 0x6e, 0x65, 0x64, 0x48, 0x65, 0x61, 0x64, 0x65, 0x72, 0x73, 0x3d, 0x2c, 0x20, 0x41, 0x6c, 0x67, 0x6f, 0x72, 0x69, 0x74,
		0x68, 0x6d, 0x3d, 0x48, 0x6d, 0x61, 0x63, 0x53, 0x48, 0x41, 0x32, 0x35, 0x36, 0x2c, 0x20, 0x53, 0x69, 0x67, 0x6e, 0x65, 0x64,
		0x3d, 0x74, 0x72, 0x75, 0x65, 0x2c, 0x20, 0x53, 0x69, 0x67, 0x6e, 0x61, 0x74, 0x75, 0x72, 0x65, 0x3d, 0x2f, 0x4b, 0x4a, 0x4f,
		0x6a, 0x59, 0x30, 0x57, 0x77, 0x48, 0x61, 0x45, 0x51, 0x57, 0x35, 0x33, 0x57, 0x75, 0x4a, 0x33, 0x4d, 0x64, 0x69, 0x78, 0x75,
		0x63, 0x35, 0x47, 0x7a, 0x73, 0x4a, 0x53, 0x55, 0x54, 0x5a, 0x68, 0x76, 0x6e, 0x58, 0x45, 0x33, 0x65, 0x77, 0x3d, 0x2c, 0x52,
		0x65, 0x6c, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x68, 0x69, 0x70, 0x56, 0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e, 0x3d, 0x30,
	}
	testValidEncodeRequestData = []byte{
		// Header
		66, 69, 0, 0, 0, 0, 3, 58, 1, 0, 1, 0, 0, 0, 0, 7,
		// Service layer
		146, 144, 146, 1, 205, 3, 51,
		// Data layer
		148, 144, 146, 1, 218, 1, 128, 153, 226, 217, 95, 24, 151, 80, 79, 129, 120, 120, 2, 60, 14, 162, 155, 70, 244,
		155, 96, 131, 103, 2, 237, 174, 100, 165, 247, 135, 57, 158, 174, 206, 191, 75, 11, 92, 200, 69, 64, 25, 56, 11,
		73, 79, 21, 179, 149, 143, 42, 118, 107, 5, 18, 121, 39, 14, 6, 155, 141, 77, 6, 40, 206, 171, 147, 184, 82, 30,
		72, 58, 94, 68, 56, 18, 231, 218, 97, 167, 225, 218, 242, 191, 222, 212, 88, 238, 192, 188, 239, 169, 119, 105,
		142, 70, 220, 132, 74, 8, 206, 230, 42, 14, 146, 234, 70, 252, 16, 9, 182, 71, 1, 97, 202, 6, 211, 121, 148,
		116, 20, 35, 131, 167, 134, 168, 238, 158, 211, 189, 179, 89, 18, 118, 166, 243, 249, 240, 149, 171, 138, 43,
		198, 252, 89, 240, 240, 97, 65, 57, 148, 71, 40, 47, 60, 205, 124, 65, 30, 185, 66, 191, 74, 100, 32, 0, 55, 59,
		18, 148, 149, 242, 174, 228, 147, 140, 37, 22, 179, 187, 173, 178, 79, 218, 120, 110, 170, 181, 215, 127, 78, 160,
		112, 172, 214, 191, 193, 143, 225, 157, 225, 222, 67, 40, 125, 25, 221, 109, 82, 88, 161, 66, 187, 216, 72, 90,
		236, 226, 218, 211, 10, 52, 204, 42, 191, 152, 198, 254, 45, 156, 53, 19, 9, 30, 142, 163, 210, 221, 230, 251,
		160, 72, 230, 167, 153, 227, 144, 120, 250, 170, 147, 48, 116, 66, 209, 36, 76, 213, 82, 240, 14, 111, 196, 92,
		81, 152, 73, 216, 38, 26, 228, 141, 89, 149, 188, 188, 99, 230, 182, 208, 28, 213, 150, 58, 249, 94, 58, 157,
		99, 248, 41, 143, 129, 101, 106, 223, 80, 198, 152, 231, 188, 169, 165, 235, 15, 135, 50, 113, 50, 253, 209,
		135, 185, 194, 87, 123, 109, 176, 244, 167, 75, 110, 243, 88, 26, 164, 139, 135, 215, 132, 253, 166, 162, 56,
		173, 172, 144, 27, 107, 228, 184, 209, 68, 60, 3, 238, 68, 2, 156, 183, 76, 243, 124, 206, 131, 4, 80, 216, 112,
		69, 145, 176, 112, 135, 209, 45, 215, 116, 191, 195, 254, 107, 65, 198, 41, 73, 61, 230, 76, 108, 181, 151, 59,
		121, 56, 146, 2, 147, 144, 146, 1, 1, 146, 2, 218, 0, 99, 99, 111, 109, 46, 97, 109, 97, 122, 111, 110, 46, 97,
		97, 97, 46, 83, 101, 115, 115, 105, 111, 110, 83, 101, 114, 118, 105, 99, 101, 86, 50, 46, 86, 83, 83, 76, 105,
		98, 101, 114, 116, 121, 69, 120, 97, 109, 112, 108, 101, 72, 122, 46, 97, 109, 122, 110, 49, 46, 97, 97, 97, 46,
		105, 100, 46, 110, 55, 122, 111, 112, 101, 117, 121, 98, 108, 113, 115, 104, 120, 106, 100, 102, 122, 116, 114,
		102, 99, 116, 122, 54, 117, 46, 68, 101, 102, 97, 117, 108, 116, 146, 4, 218, 1, 56, 65, 65, 65, 32, 105, 100,
		101, 110, 116, 105, 116, 121, 61, 99, 111, 109, 46, 97, 109, 97, 122, 111, 110, 46, 97, 97, 97, 46, 83, 101, 115,
		115, 105, 111, 110, 83, 101, 114, 118, 105, 99, 101, 86, 50, 46, 86, 83, 83, 76, 105, 98, 101, 114, 116, 121, 69,
		120, 97, 109, 112, 108, 101, 72, 122, 46, 97, 109, 122, 110, 49, 46, 97, 97, 97, 46, 105, 100, 46, 110, 55, 122,
		111, 112, 101, 117, 121, 98, 108, 113, 115, 104, 120, 106, 100, 102, 122, 116, 114, 102, 99, 116, 122, 54, 117,
		46, 68, 101, 102, 97, 117, 108, 116, 47, 49, 44, 32, 79, 112, 101, 114, 97, 116, 105, 111, 110, 61, 86, 97, 108,
		105, 100, 97, 116, 101, 83, 101, 115, 115, 105, 111, 110, 44, 32, 83, 101, 114, 118, 105, 99, 101, 61, 83, 101,
		115, 115, 105, 111, 110, 83, 101, 114, 118, 105, 99, 101, 86, 50, 44, 32, 69, 110, 99, 114, 121, 112, 116, 101,
		100, 61, 116, 114, 117, 101, 44, 32, 83, 105, 103, 110, 101, 100, 72, 101, 97, 100, 101, 114, 115, 61, 44, 32, 65,
		108, 103, 111, 114, 105, 116, 104, 109, 61, 72, 109, 97, 99, 83, 72, 65, 50, 53, 54, 44, 32, 83, 105, 103, 110,
		101, 100, 61, 116, 114, 117, 101, 44, 32, 83, 105, 103, 110, 97, 116, 117, 114, 101, 61, 70, 114, 65, 115, 83,
		120, 75, 104, 109, 73, 73, 103, 85, 116, 55, 78, 88, 87, 80, 75, 57, 105, 73, 117, 106, 104, 80, 50, 89, 119, 51,
		101, 77, 72, 77, 82, 56, 116, 78, 100, 52, 67, 107, 61, 44, 82, 101, 108, 97, 116, 105, 111, 110, 115, 104, 105,
		112, 86, 101, 114, 115, 105, 111, 110, 61, 48,
	}
)
