package tls

import (
	"crypto/x509"
	"encoding/asn1"
	"strings"

	"github.com/pkg/errors"
)

// X509 OID for AAA extension.
//
// It is undocumented extension used by AAA CA. More details can be found here:
// https://tiny.amazon.com/14vjrff7n/codeamazpackAaaTblobd682src
var aaaID = asn1.ObjectIdentifier([]int{1, 2, 840, 113549, 1, 9, 2})

// VerifyAppName returns a tls.Config.VerifyPeerCertificate compatible function
// that checks for a valid AAA APPNAME record.
//
// It reads X509 certificate extension 1.2.840.113549.1.9.2 then extracts
// APPNAME entry. TLS handshake will fail if there is no exact match of APPNAME
// found in validAppNames.
//
// This should be used by a server to verify the authenticity of calling client.
func VerifyAppName(validAppNames []string) func(rawCerts [][]byte, verifiedChains [][]*x509.Certificate) error {
	return VerifyAAACertFunc(func(certExt AAACertExtension) error {
		for _, validAppName := range validAppNames {
			if certExt.AppName == validAppName {
				return nil
			}
		}
		return errors.Errorf("invalid app name \"%s\"", certExt.AppName)
	})
}

// VerifyServiceName returns a tls.Config.VerifyPeerCertificate compatible
// function that checks for a valid AAA SERVICE_NAMES record.
//
// It reads X509 certificate extension 1.2.840.113549.1.9.2 then extracts
// SERVICE_NAMES entry. TLS handshake will fail if SERVICE_NAMES does not
// contain expectedServiceName.
//
// This should be used by a client to verify the authenticity of target service.
func VerifyServiceName(expectedServiceName string) func(rawCerts [][]byte, verifiedChains [][]*x509.Certificate) error {
	return VerifyAAACertFunc(func(certExt AAACertExtension) error {
		for _, serviceName := range certExt.ServiceNames {
			if expectedServiceName == serviceName {
				return nil
			}
		}
		return errors.Errorf("unexpected remote service name, expecting %s got %v", expectedServiceName, certExt.ServiceNames)
	})
}

// VerifyAAACertFunc returns a tls.Config.VerifyPeerCertificate compatible
// function that calls predicate function to verify the given certificate.
//
// It reads X509 certificate extension 1.2.840.113549.1.9.2 then extracts its
// fields. The predicate function will be called with the extracted fields
// as its argument. TLS handshake will fail if predicate function returns
// error.
func VerifyAAACertFunc(predicate func(certExt AAACertExtension) error) func(rawCerts [][]byte, verifiedChains [][]*x509.Certificate) error {
	return func(rawCerts [][]byte, verifiedChains [][]*x509.Certificate) error {
		certExt, err := parseAAACertExt(rawCerts[0])
		if err != nil {
			return err
		}
		return predicate(certExt)
	}
}

// AAACertExtension contains extracted fields in AAA certificate.
//
// All these fields are not publicly documented, their details below are written
// from observation.
type AAACertExtension struct {
	// AAAID is a certificate ID.
	AAAID string

	// IdentityType will be "ApolloEnvironment" for MAWS or"AwsPrincipal" for
	// Native AWS. See https://w.amazon.com/index.php/AAA/Internal/Developers/Projects/AAA%20TLS%20NAWS
	// for more information.
	IdentityType string

	// AppName is a AAA application name as seen in aaa.amazon.com.
	AppName string

	// ReadableIdentity is a combination of other fields intended for human.
	ReadableIdentity string

	// SecurityDomain is AAA security domain. It can be either "Test" or "Prod".
	SecurityDomain string

	// SecurityRegion is AAA security region. The only observed value is
	// "Default".
	SecurityRegion string

	// ServiceNames is a AAA service name as seen in aaa.amazon.com.
	ServiceNames []string

	// EnvName is the name of Apollo environment that hosts this certificate e.g.,
	// "AaaTlsShim/nobody".
	EnvName string

	// EnvStage is the stage of Apollo environment that hosts this certificate
	// e.g., "Gamma".
	EnvStage string
}

const (
	certFieldAAAID            = "AAAID"
	certFieldIdentityType     = "IDENTITY_TYPE"
	certFieldAppName          = "APPNAME"
	certFieldServiceNames     = "SERVICE_NAMES"
	certFieldReadableIdentity = "READABLE_IDENTITY"
	certFieldSecurityDomain   = "SECURITY_DOMAIN"
	certFieldSecurityRegion   = "SECURITY_REGION"
	certFieldEnvironmentName  = "ENVIRONMENT_NAME"
	certFieldEnvironmentStage = "ENVIRONMENT_STAGE"
)

func parseAAACertExt(raw []byte) (AAACertExtension, error) {
	cert, err := x509.ParseCertificate(raw)
	if err != nil {
		return AAACertExtension{}, errors.Wrap(err, "unable to parse x509 certificate")
	}

	for _, e := range cert.Extensions {
		if e.Id.Equal(aaaID) {
			var aaaID string
			_, err = asn1.Unmarshal(e.Value, &aaaID)
			if err != nil {
				return AAACertExtension{}, errors.Wrap(err, "unable to unmarshal AAA extension")
			}

			ext := AAACertExtension{}
			pairs := strings.Split(aaaID, ";")
			for _, pair := range pairs {
				kv := strings.SplitN(pair, "=", 2)
				if len(kv) == 2 {
					switch kv[0] {
					case certFieldAAAID:
						ext.AAAID = kv[1]
					case certFieldIdentityType:
						ext.IdentityType = kv[1]
					case certFieldAppName:
						ext.AppName = kv[1]
					case certFieldServiceNames:
						ext.ServiceNames = strings.Split(kv[1], "&")
					case certFieldReadableIdentity:
						ext.ReadableIdentity = kv[1]
					case certFieldSecurityDomain:
						ext.SecurityDomain = kv[1]
					case certFieldSecurityRegion:
						ext.SecurityRegion = kv[1]
					case certFieldEnvironmentName:
						ext.EnvName = kv[1]
					case certFieldEnvironmentStage:
						ext.EnvStage = kv[1]
					default:
						// Unknown key will be ignored, AAA may add more keys without
						// consulting this project.
					}
				}
			}

			return ext, nil
		}
	}

	return AAACertExtension{}, errors.Errorf("no AAA extension %s found, this is not AAA certificate", aaaID)
}
