package tls

import (
	"encoding/pem"
	"errors"
	"fmt"
	"testing"
)

const validClientCert = `-----BEGIN CERTIFICATE-----
MIIFlTCCBH2gAwIBAgIUHhWNQM70Xa00ZSfFAAABZBlYmj0wDQYJKoZIhvcNAQEL
BQAwgbUxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQH
EwdTZWF0dGxlMRMwEQYDVQQKEwpBbWF6b24uY29tMR4wHAYDVQQLExVDZXJ0aWZp
Y2F0ZSBBdXRob3JpdHkxJjAkBgNVBAMTHUFBQSBDQSBTaWduZXIgSW50ZXJtZWRp
YXRlIENBMSIwIAYJKoZIhvcNAQkBFhNhaGktdGVhbUBhbWF6b24uY29tMB4XDTE4
MDYxOTE4Mzk1NFoXDTE4MDYyNjE4Mzk1NFowUDETMBEGA1UECgwKQW1hem9uLmNv
bTE5MDcGA1UEAwwwZGV2LWRzay1jaGFpd2F0LTJhLWYyZmVjMGRiLnVzLXdlc3Qt
Mi5hbWF6b24uY29tMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAu/Bg
Naqnti4iDA3Fuer2Wgv16RYINybGiXJlE35b/IwMhRRgoAQSoT37MP5Aq5i12z/o
/a3NIieeeJzcs9z1SdIGMMXVdRWSWjkzGhOrrn6NW/Iat/Yf46Y0MtTjwX5rV+Vz
u/7/b27TSaFsEyoV8l0Brwna9Pn/nG4KDoI2a5Z17oW9K5mhOJICkNFTWYJkcVFS
igWnQ/YIGR734WPea1csNooPJEOJJBCNkEV6a32mTrAMA4NgZaScfTfuHjgxmEFi
LdhWiyf9xvz2r1y6dmmAnNx9t/k5j8HmKUKVdjfPhobqfeMYBJBNpWEG1YKwx9jt
+6Rw8yCS6bBaWfLlCwIDAQABo4IB/zCCAfswOwYDVR0RBDQwMoIwZGV2LWRzay1j
aGFpd2F0LTJhLWYyZmVjMGRiLnVzLXdlc3QtMi5hbWF6b24uY29tMB8GA1UdIwQY
MBaAFFYoKOMl6B6fhC5EAoYurYIORUCuMB0GA1UdDgQWBBQeZCT/pFCq6ynBHr3f
JfDtYQqqTzCCAS0GCSqGSIb3DQEJAgSCAR4MggEaQUFBSUQ9YW16bjEuYWFhLmlk
LjZrcXJzbmxqcXM1a2F4ZnZna2lraHR1Y3k0O0lERU5USVRZX1RZUEU9QXBvbGxv
RW52aXJvbm1lbnQ7QVBQTkFNRT1TYWJsZUF3c0Jhc2VsaW5lcjtSRUFEQUJMRV9J
REVOVElUWT1BcG9sbG9FbnY6QWFhVGxzU2hpbS9TYWJsZUF3c0Jhc2VsaW5lci9B
bHBoYTtTRUNVUklUWV9ET01BSU49VGVzdDtTRUNVUklUWV9SRUdJT049RGVmYXVs
dDtFTlZJUk9OTUVOVF9OQU1FPUFhYVRsc1NoaW0vU2FibGVBd3NCYXNlbGluZXI7
RU5WSVJPTk1FTlRfU1RBR0U9QWxwaGE7MAwGA1UdEwEB/wQCMAAwPQYDVR0fBDYw
NDAyoDCgLoYsaHR0cDovL2FoaS1jcmwuYW1hem9uLmNvbS9jcmxzL0FBQVByb2RD
QS5jcmwwDQYJKoZIhvcNAQELBQADggEBABCzwMZ3R8APtXKmexO2W2r7YLc4O/J3
sG44BQaHhbPQs4URkqqEQ+FjHyCKJbCplu50tU48u7/qnrqBdTSTp2htE8f3XkMO
+hyNMmwChoDdmgRYk8APpgk4wpTQ+m6I4mURsPmrNot1za5ziUq+uX8ZAFp3XD44
70P2+ImjZRs0mUmIHm7FacIG6c3LXUs9gorGFonenjgyWWP8VUOLQDlgSsRyhY8e
JOSYW8x5Mo139MnpQuDpqQHHDseTznKT1u/NlTNVLGm0f3R7tAuz3jFpQZptRncL
eWOmjZS+No3sWcCjp5CI14Av2lHGocGrh2sq0mwOZDfP+LWQfp0p+PU=
-----END CERTIFICATE-----`

func TestVerifyServiceName(t *testing.T) {
	tests := []struct {
		name         string
		cert         []byte
		expectedName string
		expectedErr  error
	}{
		{"valid", []byte(validPublicCert), "Partfinder", nil},
		{"no-match", []byte(validPublicCert), "BadName", errors.New("unexpected remote service name")},
		{"non-aaa", []byte(validCert), "Partfinder", errors.New("this is not AAA certificate")},
		{"client", []byte(validClientCert), "Partfinder", errors.New("unexpected remote service name")},
	}

	for _, test := range tests {
		t.Run(test.name, func(t *testing.T) {
			cert := decodeCertificate(t, test.cert)
			err := VerifyServiceName(test.expectedName)([][]byte{cert}, nil)
			if !doErrorsMatch(test.expectedErr, err) {
				t.Errorf("Expected: %v, got: %v", test.expectedErr, err)
			}
		})
	}
}

func TestVerifyAppName(t *testing.T) {
	tests := []struct {
		name         string
		cert         []byte
		expectedName string
		expectedErr  error
	}{
		{"server", []byte(validPublicCert), "A9VSHardlinesPartFinder", nil}, // server cert also has APPNAME
		{"non-aaa", []byte(validCert), "", errors.New("this is not AAA certificate")},
		{"no-match", []byte(validPublicCert), "BadName", errors.New("invalid app name")},
	}

	for _, test := range tests {
		t.Run(test.name, func(t *testing.T) {
			cert := decodeCertificate(t, test.cert)
			err := VerifyAppName([]string{test.expectedName})([][]byte{cert}, nil)
			if !doErrorsMatch(test.expectedErr, err) {
				t.Errorf("Expected: %v, got: %v", test.expectedErr, err)
			}
		})
	}
}

func TestVerifyAAACertFunc(t *testing.T) {
	tests := []struct {
		name        string
		cert        []byte
		expectedErr error
	}{
		{"non-prod", []byte(validPublicCert), errors.New("not connecting from Prod")},
		{"non-aaa", []byte(validCert), errors.New("this is not AAA certificate")},
	}

	for _, test := range tests {
		t.Run(test.name, func(t *testing.T) {
			cert := decodeCertificate(t, test.cert)
			err := VerifyAAACertFunc(func(certExt AAACertExtension) error {
				if certExt.SecurityDomain != "Prod" {
					return fmt.Errorf("client %s is not connecting from Prod", certExt.ReadableIdentity)
				}
				return nil
			})([][]byte{cert}, nil)
			if !doErrorsMatch(test.expectedErr, err) {
				t.Errorf("Expected: %v, got: %v", test.expectedErr, err)
			}
		})
	}
}

func decodeCertificate(t *testing.T, pemCert []byte) []byte {
	t.Helper()

	var block *pem.Block
	for {
		block, pemCert = pem.Decode(pemCert)
		if block == nil {
			break
		}
		if block.Type == "CERTIFICATE" {
			break
		}
	}
	if block == nil {
		t.Fatal("no certificate found in the provided PEM")
	}
	return block.Bytes
}
