package tls

import (
	"errors"
	"fmt"
	"math"
	"math/rand"
	"strings"
	"sync"
	"testing"
	"time"
)

const validCert = `-----BEGIN CERTIFICATE-----
MIID0zCCArugAwIBAgIBATANBgkqhkiG9w0BAQsFADBwMQswCQYDVQQGEwJVUzET
MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHU2VhdHRsZTETMBEGA1UEChMK
QW1hem9uLmNvbTEMMAoGA1UECxMDQUFBMRcwFQYDVQQDEw5BQUEgUm9vdCBDQSB2
MTAeFw0xNTA3MDkwMDI3NDhaFw0yNTA3MDYwMDI3NDhaMIG1MQswCQYDVQQGEwJV
UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHU2VhdHRsZTETMBEGA1UE
ChMKQW1hem9uLmNvbTEeMBwGA1UECxMVQ2VydGlmaWNhdGUgQXV0aG9yaXR5MSYw
JAYDVQQDEx1BQUEgQ0EgU2lnbmVyIEludGVybWVkaWF0ZSBDQTEiMCAGCSqGSIb3
DQEJARYTYWhpLXRlYW1AYW1hem9uLmNvbTCCASIwDQYJKoZIhvcNAQEBBQADggEP
ADCCAQoCggEBAMUlLRaCUL70drb1/A3YKos62LYsoo1gi+0QWq1hAx8WU38f/rnK
6othwn7sis2A/Md0jCiqe6BrDAraTSAN18Ao8tadgIU5RRMUApMlwvfJJanxC20A
SWIPDraVTxTr/I34RAZZbs5W/Hg6o66ms4mODqaxtqJhQQQ6ewk1145fybmbUWB1
mQMdHptF/1OTvTxr8gWiawsuZ7Q3Cnxf0Pmz0W7pkXiLJkiOjgMAF1/13ZMHILuI
N6aN/tJWlAO4AVs2Fo7nsalhF5SWuNQm1JYmd/74JkS48Soj61ovAMNqsbsjvIGo
o1AxdfDRW2wUoxacKCZkCs1MFrOncIkKmokCAwEAAaMyMDAwDwYDVR0TAQH/BAUw
AwEB/zAdBgNVHQ4EFgQUVigo4yXoHp+ELkQChi6tgg5FQK4wDQYJKoZIhvcNAQEL
BQADggEBAADxiz78/SiLCMOo5tlB76zfvFaETdNwboRsqUigfi/dJNHL9alRfTpc
Lbi4MBcBpx3Ez9iGJ1M27+0oRds5AW8nlHercenIeJuDKITNAw2PNn3lguA3SLXV
fryqJjsFY81/4+kuzzHCCL0n8U7PnP10WgWYjCBN8fxc2RfoVhpz6FuFLvUW3eJs
htnX+XMxlc+zjX81AN/5z8v6EYT+880DfLxKXLhZrz5nL2BuhMbCT1VcLP0WB3T/
hYgxzI/49SjU3zuqyCGs9cR63O2+Yt7QDY3vgwJe/zyL5wh8elGk3DeHYguu13Rb
qbYkLnfntCYfuCT+vp1c0CQ5GCS5Wm8=
-----END CERTIFICATE-----`

const invalidCert = `hello bob`

const keyPath = `/road/to/nowhere`

const validKey = `-----BEGIN RSA PRIVATE KEY-----
MIIEpAIBAAKCAQEAwh1xvoMmmoTOS7CyDCHlimfPHdAdadF35dwB5RqMc8/3csJS
RgHXIiC2kjDepJWPUYB6uCJrPozGxLMZOdMw468ombNoSY+yH4lWjb4wnpz7u9PW
P6F6ZlHAzALjyO3pcm+xhAI/pUN+8O3WXtgaEpKgyT/jt4KtRVyZKOuBFYifJY5e
DGNtJM3O08nhZZ3gbOu6wvu+MOMLv7yeTPxbc0b+nEuJ6OhSW265NnqtAnlVPJyx
6k+IZ4zR/mWFqqNXMtCg9/O9tKXG3uFw0eNfJd24Lye5+WIIarJXhb5MNDQAky8o
Pe4+7gUOl1AUOSRLoxaV68uRISr5uLztHyZPUwIDAQABAoIBAG8s6/tgJBj1nS2u
mprmQxA9GluJ2X7EsLXehttPlEe0QZXhg7uCzFNIhyg9gxaibf4Q5rhVRjUB7cm0
hesfcO3GYSt+nGkdOrhjMUZnOnNtl6Pg1OGGu5KkmhZ+RTXMRRr8q5/epdMiUC7v
GQsjG/OtNRduVx7AYjrKqRHNqBXLNJDgqU6rULFsQRkEei0g/5lipbohV8K+b3Bi
7ddmPQnEtPYaQAW62m6vGlaS5mCDKQ+yt8/1cMxZhAjiR6q5NepepyA0qWs9TUzM
6LUXLbx1a1b38DIo3Wy1b02JQTa24D0G/7A8pxeAV5NOWc1IfAgi96RLtQBg8HXM
bOtjUHkCgYEA8oiWzXbxz2of3TiucgJpWOxAkeYnhjm3cLhCHv+YIMsUdABPOlRy
p2ZUxekiiTPjBmNTqZ59z5RQUmveyW+xSSKOvT+r33gy1Fyj86C4jVgORFmKPtJd
ZpqFr35E+od0t2WqDlL1kAU++Soncnu0BSuGjU4S9Gtlf9vE1edddt0CgYEAzOSg
W/FC40VSqgSlvGC1MHdUdY01U6mDVzdXQxVo8h+Btp19zpuwpE+P68ZT91e2YwRy
Fk6pXHBTgCY8/YxafWalcSyBxmgbj9o2vYOqmVySZEZRmObi6lRcEi45Gc2X2ee/
Ixpq/T8G83h0Ib00aU0H/1hzqCkmnKKFGHQiw+8CgYEAnHdSEquxxMYKWIXAxPyZ
SizgWz/IL+f8WFQHFEiAtGdPgqbQdikU7rot5qBJD4rR48UgrET7lXdxejrtN7c/
AJeyzpTNJRY+dJJvkzUU2tr8GUxKy8ybRR5bMKSdi6LnwYls9GDA23EG92gimtED
inJSE8dSoyzGBn6ZtgXVXmECgYEAtCalVJH55Mh0GWdCrjI9vaJ+UWlaJPhbshjH
kgVGI/VN4Nny3vGHS8Fk4sKb21N02vTytuhtAwkEch2s7KY2fiH9f1fJt0CcU0xj
axhUrMl3MdGeNxlzFOQdnCKmNCjPxll8dono3khHfVVAjtXEOn3vdTU5Y/biq0A+
VDTwyGMCgYACHf+2npit+r68EhxV8tamux5Oq7yLsb271hgRcZAAfHTqE7U6ZhmQ
IBF0XCYqeWb1LvtT+sTHYePKs24SmA8HIXyyPyTw6P0RmKlkofh5ZY+fUFg0fb5V
qQvH6CaYLvWcytf4sMNlhsgVIcOmd53KrnWrOS86ZQhrf8/XsWgV3Q==
-----END RSA PRIVATE KEY-----`

const validPublicCert = `-----BEGIN CERTIFICATE-----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-----END CERTIFICATE-----`

const validKeyPair = `-----BEGIN RSA PRIVATE KEY-----
MIIEpAIBAAKCAQEAwh1xvoMmmoTOS7CyDCHlimfPHdAdadF35dwB5RqMc8/3csJS
RgHXIiC2kjDepJWPUYB6uCJrPozGxLMZOdMw468ombNoSY+yH4lWjb4wnpz7u9PW
P6F6ZlHAzALjyO3pcm+xhAI/pUN+8O3WXtgaEpKgyT/jt4KtRVyZKOuBFYifJY5e
DGNtJM3O08nhZZ3gbOu6wvu+MOMLv7yeTPxbc0b+nEuJ6OhSW265NnqtAnlVPJyx
6k+IZ4zR/mWFqqNXMtCg9/O9tKXG3uFw0eNfJd24Lye5+WIIarJXhb5MNDQAky8o
Pe4+7gUOl1AUOSRLoxaV68uRISr5uLztHyZPUwIDAQABAoIBAG8s6/tgJBj1nS2u
mprmQxA9GluJ2X7EsLXehttPlEe0QZXhg7uCzFNIhyg9gxaibf4Q5rhVRjUB7cm0
hesfcO3GYSt+nGkdOrhjMUZnOnNtl6Pg1OGGu5KkmhZ+RTXMRRr8q5/epdMiUC7v
GQsjG/OtNRduVx7AYjrKqRHNqBXLNJDgqU6rULFsQRkEei0g/5lipbohV8K+b3Bi
7ddmPQnEtPYaQAW62m6vGlaS5mCDKQ+yt8/1cMxZhAjiR6q5NepepyA0qWs9TUzM
6LUXLbx1a1b38DIo3Wy1b02JQTa24D0G/7A8pxeAV5NOWc1IfAgi96RLtQBg8HXM
bOtjUHkCgYEA8oiWzXbxz2of3TiucgJpWOxAkeYnhjm3cLhCHv+YIMsUdABPOlRy
p2ZUxekiiTPjBmNTqZ59z5RQUmveyW+xSSKOvT+r33gy1Fyj86C4jVgORFmKPtJd
ZpqFr35E+od0t2WqDlL1kAU++Soncnu0BSuGjU4S9Gtlf9vE1edddt0CgYEAzOSg
W/FC40VSqgSlvGC1MHdUdY01U6mDVzdXQxVo8h+Btp19zpuwpE+P68ZT91e2YwRy
Fk6pXHBTgCY8/YxafWalcSyBxmgbj9o2vYOqmVySZEZRmObi6lRcEi45Gc2X2ee/
Ixpq/T8G83h0Ib00aU0H/1hzqCkmnKKFGHQiw+8CgYEAnHdSEquxxMYKWIXAxPyZ
SizgWz/IL+f8WFQHFEiAtGdPgqbQdikU7rot5qBJD4rR48UgrET7lXdxejrtN7c/
AJeyzpTNJRY+dJJvkzUU2tr8GUxKy8ybRR5bMKSdi6LnwYls9GDA23EG92gimtED
inJSE8dSoyzGBn6ZtgXVXmECgYEAtCalVJH55Mh0GWdCrjI9vaJ+UWlaJPhbshjH
kgVGI/VN4Nny3vGHS8Fk4sKb21N02vTytuhtAwkEch2s7KY2fiH9f1fJt0CcU0xj
axhUrMl3MdGeNxlzFOQdnCKmNCjPxll8dono3khHfVVAjtXEOn3vdTU5Y/biq0A+
VDTwyGMCgYACHf+2npit+r68EhxV8tamux5Oq7yLsb271hgRcZAAfHTqE7U6ZhmQ
IBF0XCYqeWb1LvtT+sTHYePKs24SmA8HIXyyPyTw6P0RmKlkofh5ZY+fUFg0fb5V
qQvH6CaYLvWcytf4sMNlhsgVIcOmd53KrnWrOS86ZQhrf8/XsWgV3Q==
-----END RSA PRIVATE KEY-----
-----BEGIN CERTIFICATE-----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-----END CERTIFICATE-----`

type readFileFunc = func(string) ([]byte, error)

// chainReaders will replace the function at the location with the supplied
// replacements. After each replacement is used, the next is substituted in
// its place. Once the final replacement is used, the original function is
// restored and no other substitutions will take place.
func chainReaders(originalFunc *readFileFunc, replacements ...readFileFunc) {
	if len(replacements) == 0 {
		return
	}

	original := *originalFunc
	*originalFunc = func(s string) ([]byte, error) {
		bytes, err := replacements[0](s)
		replacements = replacements[1:]
		if len(replacements) == 0 {
			*originalFunc = original
		}

		return bytes, err
	}
}

func TestChainReaders(t *testing.T) {
	// Test that ioutilReadFile is initial
	_, err := ioutilReadFile("")
	if err == nil {
		t.Error("ioutilReadFile is already in a bad state")
	}
	originalErrMsg := err.Error()

	// Test empty replacement
	chainReaders(&ioutilReadFile)
	_, err = ioutilReadFile("")
	if err == nil {
		t.Error("chainReaders failed with empty replacements")
	}

	// Test replacements
	chainReaders(&ioutilReadFile, loadValidCert, readfileError, loadInvalidCert)
	data, err := ioutilReadFile("")
	if err != nil && string(data) != validCert {
		t.Error("chainReaders failed with a replacement")
	}
	data, err = ioutilReadFile("")
	if err == nil || !strings.Contains(err.Error(), "can't open") {
		t.Error("chainReaders failed with a replacement")
	}
	data, err = ioutilReadFile("")
	if err != nil && string(data) != invalidCert {
		t.Error("chainReaders failed with a replacement")
	}

	// Should be done, test restored
	_, err = ioutilReadFile("")
	if err.Error() != originalErrMsg {
		t.Error("chainReaders did not restore the original function")
	}
}

func TestLoadCert(t *testing.T) {
	tests := []struct {
		name        string
		readers     []readFileFunc
		expectedErr error
	}{
		{"fail key pair, fail public cert", []readFileFunc{readfileError, readfileError}, errors.New("var/state/aaatls/aaaic.pem")},
		{"fail key pair, load public cert, fail private key path", []readFileFunc{readfileError, loadValidPublicCert, readfileError}, errors.New("var/state/aaatls/key.uuid")},
		{"fail key pair, load public cert, fail private key", []readFileFunc{readfileError, loadValidPublicCert, loadKeyPath, readfileError}, errors.New(keyPath)},
		{"fail key pair, load public cert, invalid private key", []readFileFunc{readfileError, loadValidPublicCert, loadKeyPath, loadInvalidCert}, errors.New("tls: failed to find any PEM data in key input")},
		{"fail key pair, invalid public cert, load private key", []readFileFunc{readfileError, loadInvalidCert, loadKeyPath, loadValidKey}, errors.New("tls: failed to find any PEM data in certificate input")},
		{"fail key pair, load public cert, load private key, load signing cert", []readFileFunc{readfileError, loadValidPublicCert, loadKeyPath, loadValidKey, loadValidCert}, nil},
		{"invalid key pair", []readFileFunc{loadInvalidCert}, errors.New("tls: failed to find any PEM data in certificate input")},
		{"load key pair, fail signing cert", []readFileFunc{loadValidKeyPair, readfileError}, errors.New("var/state/aaatls/aaaca.pem")},
		{"load key pair, invalid signing cert", []readFileFunc{loadValidKeyPair, loadInvalidCert}, errors.New("failed to parse certificate PEM")},
		{"load key pair, load signing cert", []readFileFunc{loadValidKeyPair, loadValidCert}, nil},
	}

	for _, test := range tests {
		chainReaders(&ioutilReadFile, test.readers...)

		cp := expiringCertPool{}
		_, err := cp.loadCert()

		if !doErrorsMatch(test.expectedErr, err) {
			t.Errorf("%v: expected: %v, got: %v", test.name, test.expectedErr, err)
		}
	}
}

func TestNewExpiringCert(t *testing.T) {
	// Test error condition first
	chainReaders(&ioutilReadFile, loadValidKeyPair, loadInvalidCert)
	cp := expiringCertPool{}
	_, err := cp.newExpiringCert()
	if err == nil {
		t.Error("Expected an error type")
	}

	chainReaders(&ioutilReadFile, loadValidKeyPair, loadValidCert)
	cert, err := cp.newExpiringCert()
	if err != nil {
		t.Error("Expected an expiringCert")
	}

	current := time.Now()
	if cert.expirationTime.Before(current.Add(23*time.Hour)) || cert.expirationTime.After(current.Add(25*time.Hour)) {
		t.Error("Expected expiration time to be between 23 and 25 hours from now")
	}
}

func TestGetValidCertificate(t *testing.T) {
	tests := []struct {
		name        string
		readers     []readFileFunc
		certs       []expiringCert
		expectedErr error
	}{
		{"new certificate", []readFileFunc{loadValidKeyPair, loadValidCert}, nil, nil},
		{"new certificate with < 3 errors", []readFileFunc{readfileError, readfileError, loadValidKeyPair, loadValidCert}, nil, nil},
		{"new certificate (and fail) with ≥ 3 errors", []readFileFunc{readfileError, readfileError, readfileError, readfileError, readfileError, readfileError}, nil, errors.New("var/state/aaatls/aaaic.pem")},
		{"pooled certificate that is valid", nil, []expiringCert{{expirationTime: time.Now().AddDate(1, 0, 0)}}, nil},
		{"pooled certificate that is valid after several invalid certificates", nil, []expiringCert{{expirationTime: time.Now().AddDate(1, 0, 0)}, {expirationTime: time.Now()}, {expirationTime: time.Now()}}, nil},
		{"several pooled certificates that are invalid then a valid new certificate", []readFileFunc{loadValidKeyPair, loadValidCert}, []expiringCert{{expirationTime: time.Now()}, {expirationTime: time.Now()}}, nil},
		{"several pooled certificates that are invalid then error getting a new certificate", []readFileFunc{loadValidKeyPair, readfileError, loadValidKeyPair, readfileError, loadValidKeyPair, readfileError}, []expiringCert{{expirationTime: time.Now()}, {expirationTime: time.Now()}}, errors.New("var/state/aaatls/aaaca.pem")},
	}

	for _, test := range tests {
		chainReaders(&ioutilReadFile, test.readers...)

		cp := expiringCertPool{pool: test.certs}
		_, err := cp.getValidCertificate()

		if !doErrorsMatch(test.expectedErr, err) {
			t.Errorf("%v: expected: %v, got: %v", test.name, test.expectedErr, err)
		}
	}
}

func TestGetValidCertificateRace(t *testing.T) {

	// Test cases:
	// - Get certificate from 100 concurrent goroutines with randomly failing reader

	origReadFile := ioutilReadFile
	failPercent := 0.25
	ioutilReadFile = randomlyFailCert(failPercent)

	cp := expiringCertPool{}
	numberOfCallers := 1000

	successCh := make(chan bool, numberOfCallers)
	wg := sync.WaitGroup{}
	wg.Add(numberOfCallers)
	for i := 0; i < numberOfCallers; i++ {
		go func() {
			cert, err := cp.getValidCertificate()
			successCh <- cert != nil && err == nil
			wg.Done()
		}()
	}
	wg.Wait()
	close(successCh)

	successCount := 0
	for success := range successCh {
		if success {
			successCount++
		}
	}

	successPercent := float64(successCount) / float64(numberOfCallers)
	floorSuccessPercent := 1 - (failPercent * failPercent * failPercent) - 0.01 // Add a 1% buffer
	if successPercent < floorSuccessPercent {
		// If we failed more than expected (statistically)
		t.Errorf("Expected success: %02.1f%%, got: %02.1f%%", floorSuccessPercent*100, successPercent*100)
	}

	ioutilReadFile = origReadFile
}

func TestLoadCaCert(t *testing.T) {
	tests := []struct {
		reader      readFileFunc
		expectedErr error
	}{
		{readfileError, errors.New("/path/to/nowhere/var/state/aaatls/aaaca.pem")},
		{loadValidCert, nil},
		{loadInvalidCert, errors.New("cannot add certificate to pool")},
	}

	for _, test := range tests {
		origReadFile := ioutilReadFile
		ioutilReadFile = test.reader

		_, err := loadCaCert("/path/to/nowhere")
		if !doErrorsMatch(test.expectedErr, err) {
			t.Errorf("Expected: %v, got: %v", test.expectedErr, err)
		}

		ioutilReadFile = origReadFile
	}
}

func doErrorsMatch(expected error, got error) bool {
	if expected == nil && got == nil {
		return true
	}

	if (expected == nil) != (got == nil) {
		return false
	}

	return strings.Contains(got.Error(), expected.Error())
}

func readfileError(filename string) ([]byte, error) {
	return nil, fmt.Errorf("can't open %s", filename)
}

// randomlyFailCert will return a function that returns percent errors
func randomlyFailCert(percent float64) func(filename string) ([]byte, error) {
	n := int(math.Round(1 / percent))
	return func(filename string) ([]byte, error) {
		if r := rand.Intn(n); r == 0 {
			return nil, fmt.Errorf("can't open %s", filename)
		}

		if strings.Contains(filename, "aaaic_and_key.pem") {
			return []byte(validKeyPair), nil
		} else if strings.Contains(filename, "aaaca.pem") {
			return []byte(validCert), nil
		}

		// Default cause
		return nil, fmt.Errorf("can't open %s", filename)
	}

}

func loadValidCert(_ string) ([]byte, error) {
	return []byte(validCert), nil
}

func loadInvalidCert(_ string) ([]byte, error) {
	return []byte(invalidCert), nil
}

func loadKeyPath(_ string) ([]byte, error) {
	return []byte(keyPath), nil
}

func loadValidKey(_ string) ([]byte, error) {
	return []byte(validKey), nil
}

func loadValidPublicCert(_ string) ([]byte, error) {
	return []byte(validPublicCert), nil
}

func loadValidKeyPair(_ string) ([]byte, error) {
	return []byte(validKeyPair), nil
}
