/*
Package odincreds is a credential provider that will retrieve credentials from Odin.

This will only retrieve Principal and Credential type credentials from Odin. By default
the credentials will expire every ten minutes. Only the material name is needed to
retrieve credentials from Odin.

	sess := session.New(&aws.Config{
		Credentials: odincreds.NewCredentials("foo.bar.qux"),
	})
	svc := s3.New(sess)
*/
package odincreds

import (
	"time"

	"odin"

	"github.com/aws/aws-sdk-go/aws/credentials"
)

// DefaultDuration is the default amount of time in minutes that the credentials
// will be valid for.
var DefaultDuration = time.Duration(10) * time.Minute

// Provider is a provider that retrieves credentials from Odin using
// the OdinLocalGoClient package.
type Provider struct {
	credentials.Expiry

	// Duration is how long before the credentials expire.
	Duration time.Duration
	// ExpiryWindow will allow the credentials to trigger refreshing prior to
	// the credentials actually expiring. This is beneficial so race conditions
	// with expiring credentials do not cause request to fail unexpectedly
	// due to ExpiredTokenException exceptions.
	//
	// So a ExpiryWindow of 10s would cause calls to IsExpired() to return true
	// 10 seconds before the credentials are actually expired.
	//
	// If ExpiryWindow is 0 or less it will be ignored.
	ExpiryWindow time.Duration
	// materialName is what is used to retrieve a specific set of credentials
	// from Odin.
	materialName string
}

// OdinCredentialOption is used for initializing odin credentials. This is used to set
// configuration options for OdinCredentialProvider.
type OdinCredentialOption func(*Provider)

// NewCredentials will return credentials with the provider being the OdinCredentialProvider.
// materialName is the material set's name that is stored within Odin.
//
// By default the credentials expire every 10 minutes. However, using functional options
// this can be overwritten.
//
//	odincreds.NewCredentials("foo.bar.qux", func(provider *odincreds.Provider) {
//		provider.Duration = 15 * time.Minute
//	})
func NewCredentials(materialName string, options ...OdinCredentialOption) *credentials.Credentials {
	p := &Provider{
		Duration:     DefaultDuration,
		materialName: materialName,
	}

	for _, option := range options {
		option(p)
	}

	return credentials.NewCredentials(p)
}

// Retrieve will use the given material set and retrieve the credential pair from Odin.
func (p *Provider) Retrieve() (credentials.Value, error) {
	principal, credential, err := odin.CredentialPair(p.materialName)
	if err == nil {
		p.SetExpiration(time.Now().Add(p.Duration), p.ExpiryWindow)
	}
	return credentials.Value{
		AccessKeyID:     principal,
		SecretAccessKey: credential,
	}, err
}
