package client

import (
	"CoralGoCodec/codec"
	"errors"
	"io"
	"strings"
	"testing"
)

type roundTripHandler func(*codec.Request, io.ReadWriter) error

type testCodec struct {
	handler roundTripHandler
}

func (c *testCodec) RoundTrip(req *codec.Request, rw io.ReadWriter) error {
	return c.handler(req, rw)
}

func forceErrorRoundTrip(req *codec.Request, rw io.ReadWriter) error {
	return errors.New("force error")
}

func forceSuccessRoundTrip(req *codec.Request, rw io.ReadWriter) error {
	return nil
}

// prepareTestClient creates a new coral client with a default connection pool
func prepareTestClient(handler roundTripHandler, opts ...Option) (*client, error) {
	d := &dialerMock{}
	c := testCodec{
		handler: handler,
	}
	return NewClient("testAssembly", "testName", d, &c, opts...).(*client), nil
}

func TestClientErrorHandling(t *testing.T) {
	testCases := []struct {
		handler         roundTripHandler
		errorExpected   bool
		expectedPoolLen int
	}{
		{forceErrorRoundTrip, true, 0},
		{forceSuccessRoundTrip, false, 1},
	}

	for _, testCase := range testCases {
		testClient, err := prepareTestClient(testCase.handler)
		if err != nil {
			t.Fatalf("failed to prepare test client: %v", err)
		}
		err = testClient.Call("this", "input", "doesn't", "matter")
		if err == nil && testCase.errorExpected {
			t.Fatal("expected the client to return an error")
		} else if err != nil && !testCase.errorExpected {
			t.Fatal("expected the client to not return an error")
		}
		poolLen := testClient.connPool.(*defaultConnectionPool).length()
		if poolLen != testCase.expectedPoolLen {
			t.Fatalf("expected the client's connection pool length to be %d but received %d", testCase.expectedPoolLen, poolLen)
		}
	}
}

func TestNewClient_applyOptions(t *testing.T) {

	expectedOptionsInvocations := 3
	actualOptionInvocations := 0
	testOptions := make([]Option, expectedOptionsInvocations)

	for i := 0; i < expectedOptionsInvocations; i++ {
		testOptions[i] = func(c *client) {
			actualOptionInvocations++
		}
	}

	_, err := prepareTestClient(forceSuccessRoundTrip, testOptions...)
	if err != nil {
		t.Fatalf("failed to prepare test client: %v", err)
	}

	if actualOptionInvocations != expectedOptionsInvocations {
		t.Errorf("NewClient didn't apply the options. Expected %d options to be applied, but %d were applied instead", expectedOptionsInvocations, actualOptionInvocations)
	}
}

func TestCall_applyRequestOptions(t *testing.T) {
	testClient, err := prepareTestClient(forceSuccessRoundTrip)
	if err != nil {
		t.Fatalf("failed to prepare test client: %v", err)
	}

	expectedRequestOptionInvocations := 3
	actualRequestOptionInvocations := 0
	requestOptions := make([]RequestOption, expectedRequestOptionInvocations)
	for i := 0; i < expectedRequestOptionInvocations; i++ {
		requestOptions[i] = func(c *codec.Request) error {
			actualRequestOptionInvocations++
			return nil
		}
	}

	testClient.Call("this", "input", "doesn't", "matter", requestOptions...)
	if actualRequestOptionInvocations != expectedRequestOptionInvocations {
		t.Errorf("Call didn't apply the request options. Expected %d request options to be applied, but %d were applied instead", expectedRequestOptionInvocations, actualRequestOptionInvocations)
	}
}

func TestCall_applyRequestOptionsFail(t *testing.T) {
	testClient, err := prepareTestClient(forceSuccessRoundTrip)
	if err != nil {
		t.Fatalf("failed to prepare test client: %v", err)
	}

	requestOptions := func(c *codec.Request) error {
		return errors.New("failed to apply request option")
	}

	err = testClient.Call("this", "input", "doesn't", "matter", requestOptions)
	if err == nil {
		t.Fatal("expected Call to return an error")
	} else {
		if !strings.Contains(err.Error(), "failed to apply request option") {
			t.Fatal("expected Call to fail applying the request option")
		}
	}
}
