package coral

import (
	"coral/shape"
	"testing"
)

const (
	namespace1     = "com.amazon.test1"
	namespace1xsd  = namespace1 + ".xsd"
	namespace2     = "com.amazon.test2"
	bogusNamespace = "com.amazon.bogus"
)

type lookupShapeResult struct {
	shapeInterface shape.Interface
	hasErr         bool
}

type testFixture struct {
	lookupRef  shape.Reference
	asm1Shapes []shape.Interface
	asm2Shapes []shape.Interface
	result     lookupShapeResult
}

func (fixture *testFixture) GetAssemblies() Assemblies {
	asm1 := Assembly{namespace1, fixture.asm1Shapes, []shape.Service{}}
	asm2 := Assembly{namespace2, fixture.asm2Shapes, []shape.Service{}}
	return Assemblies{asm1, asm2}
}

var shapeRef1 = shape.NewReference(namespace1, namespace1, "MyBoolean")
var shapeInt1 = shape.NewBoolean(shapeRef1)

var shapeRef1WithXsd = shape.NewReference(namespace1, namespace1xsd, "MyBoolean")
var shapeInt1WithXsd = shape.NewBoolean(shapeRef1WithXsd)

var shapeRef2 = shape.NewReference(namespace2, namespace2, "MyInteger")
var shapeInt2 = shape.NewInteger(shapeRef2)

var crossNamespaceShapeRef = shape.NewReference(namespace1, namespace2, "MyBoolean")
var crossNamespaceShapeInt = shape.NewBoolean(crossNamespaceShapeRef)

var lookupShapeTests = []testFixture{
	{
		shapeRef1,
		[]shape.Interface{shapeInt1},
		[]shape.Interface{shapeInt2},
		lookupShapeResult{shapeInt1, false},
	},
	{
		shape.NewReference(bogusNamespace, bogusNamespace, "MyChar"),
		[]shape.Interface{shapeInt1},
		[]shape.Interface{shapeInt2},
		lookupShapeResult{nil, true},
	},
	{
		shapeRef1WithXsd,
		[]shape.Interface{shapeInt1WithXsd},
		[]shape.Interface{shapeInt2},
		lookupShapeResult{shapeInt1WithXsd, false},
	},
	{
		crossNamespaceShapeRef,
		[]shape.Interface{},
		[]shape.Interface{crossNamespaceShapeInt, shapeInt2},
		lookupShapeResult{crossNamespaceShapeInt, false},
	},
}

func Test_LookupShape_MatchingAssemblyNamespace(t *testing.T) {
	for i, fixture := range lookupShapeTests {
		assemblies := fixture.GetAssemblies()
		result, err := assemblies.LookupShape(fixture.lookupRef)

		if fixture.result.hasErr != (err != nil) {
			t.Errorf("Test fixture #%d: LookupShape err result was incorrect. Error expected: %t. Got: %t",
				i, fixture.result.hasErr, err != nil)
		}

		if fixture.result.shapeInterface != result {
			t.Errorf("Test fixture #%d: LookupShape shape interface result was incorrect. Expected: %s. Got: %s",
				i, fixture.result.shapeInterface, result)
		}
	}
}
