package driver

import (
	"brazilpath"
	"coral/encoding"
	"coral/shape"
	"fmt"
	"io"
	"os"
	"os/exec"
	"strings"
)

func Exec(decoder encoding.Decoder, encoder encoding.Encoder, compile bool) {
	readers := []io.Reader{}
	paths, err := brazilpath.All.CoralModel()
	if err != nil {
		panic(err)
	}

	for _, dir := range paths {
		rdrs := openDir(dir)
		readers = append(readers, rdrs...)
	}

	ColorPrintln(ColorCyan, "Decoding")
	assemblies := decoder.Decode(readers)
	ColorPrintf(ColorGreen, "Finished Decoding %d Assemblies\n", len(assemblies))

	pathSep := string([]rune{os.PathSeparator})
	buildDir, err := brazilpath.PackageSrcRoot()
	if err != nil {
		panic(err)
	}
	basePath := strings.Join([]string{buildDir, "src", ""}, pathSep)
	err = os.MkdirAll(basePath, os.ModeDir|os.ModePerm)
	if err != nil {
		panic(err)
	}

	ColorPrintln(ColorCyan, "Generating Client")
	// Collect services to target.  We will target services
	// defined in any assembly in scope of the Coral generation
	targetSvcs := make([]shape.Service, 0)
	for _, asm := range assemblies {
		targetSvcs = append(targetSvcs, asm.Services...)
	}

	for _, asm := range assemblies {
		fmt.Printf("Encoding Assembly %s\n", asm.Namespace)
		for name, contents := range encoder.Encode(asm, assemblies, targetSvcs) {
			path := basePath + name
			if i := strings.LastIndex(path, pathSep); i >= 0 {
				err = os.MkdirAll(path[:i], os.ModeDir|os.ModePerm)
				if err != nil {
					panic(err)
				}
			}
			fmt.Printf("%s\n", path)
			file, err := os.Create(path)
			if err != nil {
				panic(err)
			} else {
				io.WriteString(file, contents)
			}
		}
	}
	ColorPrintln(ColorGreen, "Finished Generating Client")

	/*
		For some models the generating client is not valid Go code and won't compile due to unused imports.
		To address this goimports is invoked, if available, to clean up the generated source.
	*/
	ColorPrintln(ColorCyan, "Checking for Existence of goimports")
	// Check if goimports is available by attempting to execute it. If there is no error it's available.
	if exec.Command("goimports").Run() != nil {
		ColorPrintln(ColorYellow, "WARN: Skipping goimports as it's not available. Consider upgrading to a newer version of GoLang to automatically pick up goimports.")
	} else {
		ColorPrintln(ColorCyan, "Running goimports")
		cmd := exec.Command("goimports", "-w", "src")
		cmd.Stdout, cmd.Stderr = os.Stdout, os.Stderr
		if err = cmd.Run(); err != nil {
			panic(err)
		}
		ColorPrintln(ColorGreen, "Finished goimports Overwrite")
	}

	if !compile {
		ColorPrintln(ColorYellow, "WARN: Compile skipped")
		return
	}

	ColorPrintln(ColorCyan, "Compiling")

	cmd := exec.Command("bgo-wrap-make", "release")
	cmd.Stdout, cmd.Stderr = os.Stdout, os.Stderr
	err = cmd.Run()
	if err != nil {
		panic(err)
	}

	ColorPrintln(ColorGreen, "Finished Compiling Client")
}

func openDir(dirname string) []io.Reader {
	dir, err := os.Open(dirname)
	if err != nil {
		panic(fmt.Errorf(`os.Open("%s") = %v, %s`, dirname, dir, err))
	}

	if stat, err := dir.Stat(); err == nil {
		if !stat.IsDir() {
			err = fmt.Errorf("Something went wrong %s isn't a directory.", dir.Name())
			panic(err)
		}
	} else {
		panic(err)
	}

	// Potential optimization would be to read the last modified time of the model file
	// and the generated source. Don't generate if they are the same or the source is after the model.

	fileinfos, err := dir.Readdir(0) // >= 0 means read all
	if err != nil {
		panic(err)
	}
	readers := make([]io.Reader, 0, len(fileinfos))
	for _, info := range fileinfos {
		absPath := fmt.Sprintf("%s%c%s", dir.Name(), os.PathSeparator, info.Name())
		if info.IsDir() {
			readers = append(readers, openDir(absPath)...)
		} else if strings.HasSuffix(info.Name(), ".xml") {
			file, err := os.Open(absPath)
			if err != nil {
				panic(err)
			}
			println("Found " + absPath)
			readers = append(readers, file)
		} else {
			println("Ignored " + absPath)
		}
	}
	return readers
}

// See https://misc.flogisoft.com/bash/tip_colors_and_formatting
const (
	ColorReset  = "\033[39m"
	ColorGreen  = "\033[32m"
	ColorYellow = "\033[33m"
	ColorCyan   = "\033[36m"
)

// Consider importing https://github.com/fatih/color
func ColorPrintln(color string, a ...interface{}) {
	fmt.Println(color + fmt.Sprint(a...) + ColorReset)
}
func ColorPrintf(color, format string, a ...interface{}) {
	fmt.Printf(color+format+ColorReset, a...)
}
