/* Copyright 2016 Amazon.com, Inc. or its affiliates. All Rights Reserved. */
package golang

import (
	"go/ast"
	"go/token"
)

func generateClientAST(f fileGen) *ast.File {
	file := ast.File{
		Name:    ast.NewIdent(f.asmFile().pkg()),
		Decls:   make([]ast.Decl, 0),
		Imports: make([]*ast.ImportSpec, 0),
	}

	addType := func(name string, docs *ast.CommentGroup, expr ast.Expr) {
		decl := &ast.GenDecl{
			Doc:   docs,
			Tok:   token.TYPE,
			Specs: []ast.Spec{&ast.TypeSpec{Name: ast.NewIdent(name), Type: expr}},
		}
		file.Decls = append(file.Decls, decl)
	}

	for _, shape := range f.asmFile().asm.Shapes {
		if importSpecs := f.importSpecs(shape); importSpecs != nil {
			for _, importSpec := range importSpecs {
				addImport(&file, importSpec)
			}
		}
		if expr := f.typeExpr(shape); expr != nil {
			name := f.publicIdent(shape.Name())
			docs := comment(shape.Doc())
			addType(name, docs, expr)
		}

		if expr := f.implExpr(shape); expr != nil {
			name := f.privateIdent(shape.Name())
			if importSpecs := f.importSpecs(shape); importSpecs != nil {
				for _, importSpec := range importSpecs {
					addImport(&file, importSpec)
				}
			}
			addType(name, nil, expr)
		}

		if decls := f.implDecls(shape); decls != nil {
			file.Decls = append(file.Decls, decls...)
		}
	}

	for pkg, path := range f.asmFile().externAsms {
		spec := &ast.ImportSpec{
			Path: &ast.BasicLit{
				Kind:  token.STRING,
				Value: `"` + path + `"`,
			},
			Name: ast.NewIdent(pkg),
		}
		addImport(&file, spec)
	}

	if len(file.Imports) > 0 {
		specs := make([]ast.Spec, 0, len(file.Imports))
		for _, spec := range file.Imports {
			specs = append(specs, spec)
		}
		importDecl := &ast.GenDecl{
			Tok:    token.IMPORT,
			Specs:  specs,
			Lparen: token.Pos(1),
		}
		file.Decls = append([]ast.Decl{importDecl}, file.Decls...)
	}

	return &file
}

func generateServerAST(f fileGen) *ast.File {
	file := ast.File{
		Name:    ast.NewIdent(f.asmFile().pkg()),
		Decls:   make([]ast.Decl, 0),
		Imports: make([]*ast.ImportSpec, 0),
	}

	addType := func(name string, docs *ast.CommentGroup, expr ast.Expr) {
		decl := &ast.GenDecl{
			Doc:   docs,
			Tok:   token.TYPE,
			Specs: []ast.Spec{&ast.TypeSpec{Name: ast.NewIdent(name), Type: expr}},
		}
		file.Decls = append(file.Decls, decl)
	}

	for _, shape := range f.asmFile().asm.Shapes {
		if importSpecs := f.importSpecs(shape); importSpecs != nil {
			for _, importSpec := range importSpecs {
				addImport(&file, importSpec)
			}
		}
		if expr := f.typeExpr(shape); expr != nil {
			name := f.publicIdent(shape.Name())
			docs := comment(shape.Doc())
			addType(name, docs, expr)
		}

		if expr := f.implExpr(shape); expr != nil {
			name := f.privateIdent(shape.Name())
			if importSpecs := f.importSpecs(shape); importSpecs != nil {
				for _, importSpec := range importSpecs {
					addImport(&file, importSpec)
				}
			}
			addType(name, nil, expr)
		}

		if decls := f.implDecls(shape); decls != nil {
			file.Decls = append(file.Decls, decls...)
		}
	}

	for pkg, path := range f.asmFile().externAsms {
		spec := &ast.ImportSpec{
			Path: &ast.BasicLit{
				Kind:  token.STRING,
				Value: `"coralserver/` + path + `"`,
			},
			Name: ast.NewIdent(pkg),
		}
		addImport(&file, spec)
	}

	if len(file.Imports) > 0 {
		specs := make([]ast.Spec, 0, len(file.Imports))
		for _, spec := range file.Imports {
			specs = append(specs, spec)
		}
		importDecl := &ast.GenDecl{
			Tok:    token.IMPORT,
			Specs:  specs,
			Lparen: token.Pos(1),
		}
		file.Decls = append([]ast.Decl{importDecl}, file.Decls...)
	}

	//TODO: Sort the file to make it easier to look through

	return &file
}

func addImport(file *ast.File, importSpec *ast.ImportSpec) {
	//Make sure we haven't already imported the pkg
	found := false
	for _, spec := range file.Imports {
		if spec.Path.Value == importSpec.Path.Value {
			found = true
			break
		}
	}
	if !found {
		file.Imports = append(file.Imports, importSpec)
	}
}
