package golang

import (
	"coral/shape"
	"go/ast"
	"go/token"
)

// typeExpr satisfies fileGen.
// Generates the service client struct.
func (f *clientFile) typeExpr(s shape.Interface) ast.Expr {
	switch i := s.(type) {
	case *shape.Service:
		return f.serviceTypeExpr(i)
	}
	return nil
}

// implExpr satisfies fileGen.
// Returns nil
func (f *clientFile) implExpr(s shape.Interface) ast.Expr {
	//no impls
	return nil
}

// serviceTypeExpr generates the service client struct with the field
// C __client__.Client
func (f *clientFile) serviceTypeExpr(s *shape.Service) ast.Expr {
	return &ast.StructType{
		Fields: &ast.FieldList{
			List: []*ast.Field{
				&ast.Field{
					Names: []*ast.Ident{ast.NewIdent("C")},
					Type:  ast.NewIdent("__client__.Client"),
				},
			},
		},
	}
}

// importSpecs satisfies fileGen.
// Generates imports for coral/client, coral/codec, and coral/dialer.
func (f *clientFile) importSpecs(s shape.Interface) []*ast.ImportSpec {
	switch s.(type) {
	case *shape.Service:
		specs := []*ast.ImportSpec{
			&ast.ImportSpec{
				Name: ast.NewIdent("__client__"),
				Path: &ast.BasicLit{
					Kind:  token.STRING,
					Value: `"coral/client"`,
				},
			},
			&ast.ImportSpec{
				Name: ast.NewIdent("__codec__"),
				Path: &ast.BasicLit{
					Kind:  token.STRING,
					Value: `"coral/codec"`,
				},
			},
			&ast.ImportSpec{
				Name: ast.NewIdent("__dialer__"),
				Path: &ast.BasicLit{
					Kind:  token.STRING,
					Value: `"coral/dialer"`,
				},
			},
		}
		return specs
	}
	return nil
}

// implDecls satisfies fileGen.
// Generates function implementations.
func (f *clientFile) implDecls(s shape.Interface) []ast.Decl {
	switch i := s.(type) {
	case *shape.Service:
		return f.serviceImplDecls(i)
	}
	return nil
}

// serviceImplDecls generates a client builder and function implementations for
// each of the service operations.
func (f *clientFile) serviceImplDecls(s *shape.Service) []ast.Decl {
	decls := make([]ast.Decl, 0, len(s.Operations)+1)
	decls = append(decls, f.newServiceDecl(s))
	for _, op := range s.Operations {
		t := f.ptrIdent(s.Name())
		decls = append(decls, f.operationImplDecl(t, op))
	}
	return decls
}

// newServiceDecl generates a client builder.
func (f *clientFile) newServiceDecl(s *shape.Service) ast.Decl {
	funcType := &ast.FuncType{
		Params: fieldList(
			field("dialer", "__dialer__.Dialer"),
			field("codec", "__codec__.RoundTripper"),
			field("options", "...__client__.Option")),
		Results: fieldList(field("service", f.ptrIdent(s.Name()))),
	}

	newClient := &ast.CallExpr{
		Fun: ast.NewIdent("__client__.NewClient"),
		Args: []ast.Expr{
			ast.NewIdent(`"` + s.Name().Assembly() + `"`),
			ast.NewIdent(`"` + s.Name().Name() + `"`),
			ast.NewIdent("dialer"),
			ast.NewIdent("codec"),
			ast.NewIdent("options..."),
		},
	}

	structExpr := &ast.CompositeLit{
		Type: ast.NewIdent(f.publicIdent(s.Name())),
		Elts: []ast.Expr{newClient},
	}

	body := blockStmt(returnStmt(pointerTo(structExpr)))
	name := f.publicIdent(s.Name())
	return &ast.FuncDecl{
		Doc:  comment("Creates a new " + name),
		Name: ast.NewIdent("New" + name),
		Type: funcType,
		Body: body,
	}
}

// ptrIdent returns a string that represents a pointer to the public
// identity of the client.
func (f *clientFile) ptrIdent(r shape.Reference) string {
	return "*" + f.publicIdent(r)
}

// publicIdent satisfies fileGen.
// Returns a string that represents the public identity of the client.
func (f *clientFile) publicIdent(r shape.Reference) string {
	return f.asmFile().publicIdent(r) + "Client"
}

// operationImplDecl generates the implementation of an operation with logic in the form:
// 		var output OutputType
// 		err := this.C.Call(namespace, operation, input, &output)
// 		return output, err
// With variations based on whether the input or output is missing.
func (f *clientFile) operationImplDecl(t string, o *shape.Operation) ast.Decl {
	thisIdent := "this"
	inputIdent := "input"
	outputIdent := "output"
	errIdent := "err"
	input := &ast.FieldList{}
	output := fieldList(field("", "error"))
	if o.Input != nil {
		input.List = []*ast.Field{
			field(inputIdent, f.asmFile().publicType(o.Input)),
		}
	} else {
		inputIdent = ""
	}

	if o.Output != nil {
		newField := field("", f.asmFile().publicType(o.Output))
		output.List = append([]*ast.Field{newField}, output.List...)
	} else {
		outputIdent = ""
	}

	/*
	   body := {
	       var output NewFooOutput
	       err := this.C.Call("com.amazon.foosvc","GetFoo",input,&output)
	       return output,err
	   }
	*/

	body := blockStmt()
	// var output NewFooOutput
	if outputIdent != "" {
		stmt := declare_var(outputIdent, f.asmFile().publicType(o.Output))
		body.List = append(body.List, stmt)
	}
	//err := this.C.Call(...)
	{
		oAsmArg := &ast.BasicLit{
			Kind:  token.STRING,
			Value: `"` + f.asmFile().asm.Namespace + `"`,
		}
		oNameArg := &ast.BasicLit{
			Kind:  token.STRING,
			Value: `"` + o.Name().Name() + `"`,
		}
		var inputArg ast.Expr
		var outputArg ast.Expr
		if outputIdent == "" {
			outputArg = ast.NewIdent("nil")
		} else {
			outputArg = &ast.UnaryExpr{
				Op: token.AND,
				X:  ast.NewIdent(outputIdent),
			}
		}
		if inputIdent == "" {
			inputArg = ast.NewIdent("nil")
		} else {
			inputArg = ast.NewIdent(inputIdent)
		}
		stmt := define_var(errIdent, &ast.CallExpr{
			Fun: &ast.SelectorExpr{
				X: &ast.SelectorExpr{
					X:   ast.NewIdent(thisIdent),
					Sel: ast.NewIdent("C"),
				},
				Sel: ast.NewIdent("Call"),
			},
			Args: []ast.Expr{
				oAsmArg, oNameArg, inputArg, outputArg,
			},
		})
		body.List = append(body.List, stmt)
	}
	//return output,err
	if outputIdent != "" {
		stmt := returnStmt(
			ast.NewIdent(outputIdent),
			ast.NewIdent(errIdent),
		)
		body.List = append(body.List, stmt)
	} else {
		//return err
		stmt := returnStmt(ast.NewIdent(errIdent))
		body.List = append(body.List, stmt)
	}

	return &ast.FuncDecl{
		Doc:  comment(o.Doc()),
		Recv: fieldList(field(thisIdent, t)),
		Name: ast.NewIdent(f.asmFile().publicIdent(o.Name())),
		Type: &ast.FuncType{
			Params:  input,
			Results: output,
		},
		Body: body,
	}
}
