package golang

import (
	"bytes"
	coral "coral/assembly"
	"coral/encoding"
	"coral/shape"
	"go/ast"
	"go/format"
	"go/printer"
	"go/token"
)

//Information for encoding the entire package
type ClientEncoder struct {
	PkgName string
}

type ModelEncoder struct {
	PkgName string
}

type ServerModelEncoder struct {
	PkgName string
}

type ServerEncoder struct {
	PkgName string
}

//probably not a very good name
//Information for a single assembly encoding
type assemblyFile struct {
	asm        coral.Assembly
	allAsms    coral.Assemblies
	targetSvcs []shape.Service
	pkgName    string
	externAsms map[string]string //[pkg]path
}

func (f *assemblyFile) asmFile() *assemblyFile { return f }

type modelFile struct {
	*assemblyFile
}

type clientFile struct {
	*assemblyFile
}

type serverModelFile struct {
	*assemblyFile
}

type serverFile struct {
	*assemblyFile
}

type fileGen interface {
	typeExpr(shape.Interface) ast.Expr
	publicIdent(shape.Reference) string
	publicType(shape.Reference) ast.Expr
	importSpecs(shape.Interface) []*ast.ImportSpec
	implDecls(shape.Interface) []ast.Decl
	implExpr(shape.Interface) ast.Expr
	privateIdent(shape.Reference) string
	asmFile() *assemblyFile
}

// Assert that we fulfill the interface
var _ encoding.Encoder = ClientEncoder{}
var _ encoding.Encoder = ModelEncoder{}
var _ encoding.Encoder = ServerEncoder{}
var _ encoding.Encoder = ServerModelEncoder{}

func (e ClientEncoder) Encode(asm coral.Assembly, asms coral.Assemblies, targetSvcs []shape.Service) map[string]string {
	files := ModelEncoder{e.PkgName}.Encode(asm, asms, targetSvcs)
	f := &clientFile{&assemblyFile{asm, asms, targetSvcs, e.PkgName, map[string]string{}}}
	clientPath := f.path() + "/client.go"
	clientWriter := new(bytes.Buffer)
	fset := token.NewFileSet()
	printer.Fprint(clientWriter, fset, generateClientAST(f))
	src := clientWriter.Bytes()
	// Format the generated source
	formatted, err := format.Source(src)
	if err != nil {
		formatted = src
	}
	files[clientPath] = string(formatted)
	return files
}

func (e ModelEncoder) Encode(asm coral.Assembly, asms coral.Assemblies, targetSvcs []shape.Service) map[string]string {
	asmFile := assemblyFile{asm, asms, targetSvcs, e.PkgName, map[string]string{}}
	f := modelFile{&asmFile}
	modelPath := f.path() + "/model.go"
	modelWriter := new(bytes.Buffer)
	fset := token.NewFileSet()
	printer.Fprint(modelWriter, fset, generateClientAST(&f))
	src := modelWriter.Bytes()
	// Format the generated source
	formatted, err := format.Source(src)
	if err != nil {
		formatted = src
	}
	return map[string]string{modelPath: string(formatted)}
}

func (e ServerEncoder) Encode(asm coral.Assembly, asms coral.Assemblies, targetSvcs []shape.Service) map[string]string {
	files := ServerModelEncoder{e.PkgName}.Encode(asm, asms, targetSvcs)
	f := &serverFile{&assemblyFile{asm, asms, targetSvcs, e.PkgName, map[string]string{}}}
	serverPath := "coralserver/" + f.path() + "/generated_server.go"
	serverWriter := new(bytes.Buffer)
	fset := token.NewFileSet()
	printer.Fprint(serverWriter, fset, generateServerAST(f))
	src := serverWriter.Bytes()
	// Format the generated source
	formatted, err := format.Source(src)
	if err != nil {
		formatted = src
	}
	files[serverPath] = string(formatted)
	return files
}

func (e ServerModelEncoder) Encode(asm coral.Assembly, asms coral.Assemblies, targetSvcs []shape.Service) map[string]string {
	asmFile := assemblyFile{asm, asms, targetSvcs, e.PkgName, map[string]string{}}
	f := serverModelFile{&asmFile}
	modelPath := "coralserver/" + f.path() + "/model.go"
	modelWriter := new(bytes.Buffer)
	fset := token.NewFileSet()
	printer.Fprint(modelWriter, fset, generateServerAST(&f))
	src := modelWriter.Bytes()
	// Format the generated source
	formatted, err := format.Source(src)
	if err != nil {
		formatted = src
	}
	return map[string]string{modelPath: string(formatted)}
}
