package golang

import (
	"github.com/pkg/errors"
	"go/ast"
	"go/token"
	"strings"
)

func declare_var(name string, t interface{}) *ast.DeclStmt {
	return &ast.DeclStmt{
		Decl: &ast.GenDecl{
			Tok: token.VAR,
			Specs: []ast.Spec{
				&ast.ValueSpec{
					Names: []*ast.Ident{ast.NewIdent(name)},
					Type:  exprOf(t),
				},
			},
		},
	}
}

func exprOf(t interface{}) ast.Expr {
	if e, ok := t.(ast.Expr); ok {
		return e
	} else if s, ok := t.(string); ok {
		return ast.NewIdent(s)
	} else {
		panic(errors.Errorf("Unknown type: %T", t))
	}
}

func define_var(name string, rhs ast.Expr) *ast.AssignStmt {
	return define_vars([]string{name}, rhs)
}

func define_vars(names []string, rhs ...ast.Expr) *ast.AssignStmt {
	lhs := make([]ast.Expr, len(names))
	for i, name := range names {
		lhs[i] = ast.NewIdent(name)
	}
	return &ast.AssignStmt{
		Lhs: lhs,
		Tok: token.DEFINE,
		Rhs: rhs,
	}
}

func assign_var(name string, rhs ast.Expr) *ast.AssignStmt {
	return &ast.AssignStmt{
		Lhs: []ast.Expr{ast.NewIdent(name)},
		Tok: token.ASSIGN,
		Rhs: []ast.Expr{rhs},
	}
}

func call(lhs ast.Expr, rhs *ast.Ident) *ast.CallExpr {
	return &ast.CallExpr{
		Fun: &ast.SelectorExpr{
			X:   lhs,
			Sel: rhs,
		},
	}
}

func blockStmt(stmts ...ast.Stmt) *ast.BlockStmt {
	return &ast.BlockStmt{
		List: stmts,
	}
}

func returnStmt(exprs ...ast.Expr) *ast.ReturnStmt {
	return &ast.ReturnStmt{
		Results: exprs,
	}
}

func fieldList(fields ...*ast.Field) *ast.FieldList {
	return &ast.FieldList{
		List: fields,
	}
}

func field(name string, t interface{}) *ast.Field {
	var names []*ast.Ident = nil
	if name != "" {
		names = []*ast.Ident{ast.NewIdent(name)}
	}
	return &ast.Field{
		Type:  exprOf(t),
		Names: names,
	}
}

func pointerTo(x ast.Expr) *ast.UnaryExpr {
	return &ast.UnaryExpr{
		Op: token.AND,
		X:  x,
	}
}

// comment turns the given comment text into a CommentGroup by turning each individual
// line into a comment.  If there is no text, then nil is returned.
func comment(text string) *ast.CommentGroup {
	if text == "" {
		return nil
	}
	lines := strings.Split(text, "\n")
	group := &ast.CommentGroup{List: make([]*ast.Comment, 0, len(lines))}
	for _, line := range lines {
		line = strings.TrimSpace(line)
		group.List = append(group.List, &ast.Comment{Text: "//" + line})
	}
	return group
}
