package golang

import (
	"coral/shape"
	"github.com/pkg/errors"
	"go/ast"
	"go/token"
	"strings"
)

// implExpr returns the type expression for a private type that implements the shape
func (f *modelFile) implExpr(iface shape.Interface) ast.Expr {
	switch s := iface.(type) {
	case *shape.Structure:
		return f.structImplExpr(s)
	}
	return nil
}

// typeExpr returns the type expression that defines the shape
func (f *modelFile) typeExpr(iface shape.Interface) ast.Expr {
	if nativeType := f.nativeType(iface); nativeType != nil {
		//The type can be represented natively in Go
		return nil
	}
	switch s := iface.(type) {
	case *shape.Structure:
		return f.structTypeExpr(s)
	case *shape.Operation:
		return f.operationTypeExpr(s)
	case *shape.Service:
		return f.serviceTypeExpr(s)
	}
	panic(errors.Errorf("Bug: Unknown type %T", iface))
}

func (f *modelFile) implDecls(iface shape.Interface) []ast.Decl {
	switch s := iface.(type) {
	case *shape.Structure:
		return f.structImplDecls(s)
	case *shape.Operation, *shape.Service:
		return nil
	}
	return f.genericImplDecls(iface)
}

func (f *modelFile) genericImplDecls(s shape.Interface) []ast.Decl {
	decls := make([]ast.Decl, 0)
	fromType := f.publicType(s.Name())
	toType := f.publicType(s.Name())
	toIdent := f.publicIdent(s.Name())
	body := blockStmt()
	/*
	   func init() {
	       (1)        var val To
	       (2)        t := __reflect__.TypeOf(&val).Elem()
	       (3)        fun := func(from __reflect__.Value) reflect.Value {
	           //Get things down to their real types
	           var to To
	           if f,ok := from.(From); ok {
	               to = To(f)
	           } else {
	               panic("Can't convert")
	           }
	           return __reflect__.ValueOf(to)
	       }
	       (4)      __model__.RegisterConverter(t,fun)
	   }
	*/

	//(1)
	body.List = append(body.List, declare_var("val", toType))
	//(2)
	body.List = append(body.List, define_var("t", call(
		__reflect__.TypeOf("val"), ast.NewIdent("Elem"))),
	)
	//(3)
	body.List = append(body.List, define_var("fun", &ast.FuncLit{
		Type: &ast.FuncType{
			Results: fieldList(__reflect__.Value()),
			Params:  fieldList(__reflect__.Value("from")),
		},
		Body: blockStmt(
			declare_var("to", toType),
			&ast.IfStmt{
				Init: define_vars([]string{
					"f", "ok",
				}, &ast.TypeAssertExpr{
					X: call(
						ast.NewIdent("from"),
						ast.NewIdent("Interface"),
					),
					Type: fromType,
				},
				),
				Cond: ast.NewIdent("ok"),
				Body: blockStmt(assign_var("to", ast.NewIdent("f"))),
				Else: blockStmt(
					&ast.ExprStmt{
						//TODO: Better panic msg
						X: builtin.Panic(`"Can't Convert from " + from.Type().Name() + " to ` + toIdent + `"`),
					},
				),
			},
			returnStmt(__reflect__.ValueOf("to"))),
	}))
	//(4)
	body.List = append(body.List, &ast.ExprStmt{
		X: &ast.CallExpr{
			Fun:  ast.NewIdent("__model__.RegisterConverter"),
			Args: []ast.Expr{ast.NewIdent("t"), ast.NewIdent("fun")},
		},
	})
	fun := &ast.FuncDecl{
		Name: ast.NewIdent("init"),
		Type: &ast.FuncType{},
		Body: body,
	}
	decls = append(decls, fun)
	return decls
}

func (f *modelFile) importSpecs(iface shape.Interface) []*ast.ImportSpec {
	seen := map[shape.Interface]bool{}
	var specFunc func(iface shape.Interface) []string
	specFunc = func(iface shape.Interface) []string {
		paths := []string{}
		if _, ok := seen[iface]; ok {
			//Prevent infinite loop for circular dependencies
			return paths
		}
		seen[iface] = true
		switch iface := iface.(type) {
		case *shape.BigInteger, *shape.BigDecimal:
			paths = append(paths, "math/big")
		case *shape.Timestamp:
			paths = append(paths, "time")
		case *shape.Structure:
			for _, ref := range iface.Members {
				s, err := f.allAsms.LookupShape(ref)
				if err != nil {
					continue
				}
				if ref.Assembly() != f.asm.Namespace && !shape.IsSimple(s) {
					continue
				}
				paths = append(paths, specFunc(s)...)
			}
			paths = append(paths, "reflect", "coral/model")
		case *shape.Service:
			// some services have only references and are empty
			if len(iface.Operations) != 0 {
				paths = append(paths, "reflect", "coral/model")
			}
		default:
			paths = append(paths, "reflect", "coral/model")
		}
		return paths
	}
	paths := specFunc(iface)

	specs := make([]*ast.ImportSpec, 0, len(paths))
	for _, path := range paths {
		paths = strings.Split(path, "/")
		path = `"` + path + `"`
		specs = append(specs, &ast.ImportSpec{
			Name: ast.NewIdent("__" + paths[len(paths)-1] + "__"), //Change every import to __name__
			Path: &ast.BasicLit{
				Kind:  token.STRING,
				Value: path,
			},
		})
	}
	return specs
}

func (f *modelFile) structTypeExpr(structure *shape.Structure) ast.Expr {
	/*
	   interface {
	       [SuperType Methods]
	       Foo() Foo
	       SetFoo(Foo f)
	    }
	*/
	list := make([]*ast.Field, 0, 2*len(structure.Members)) // 2* for Getter & Setter

	if isa := structure.IsA; isa != nil {
		// emit SuperType methods
		super, err := f.allAsms.LookupShape(*isa)
		if err != nil {
			panic(err)
		}
		if super, ok := super.(*shape.Structure); ok {
			list = append(list, f.structTypeExpr(super).(*ast.InterfaceType).Methods.List...)
		}
	}

	if structure.IsException && structure.IsA == nil {
		//TODO: Verify that we only need to emit error type for base classes
		errorField := &ast.Field{
			Type: ast.NewIdent("error"),
		}
		list = append(list, errorField)
	}
	for name, member := range structure.Members {
		doc := comment(member.Doc)
		getter := &ast.Field{
			Doc:   doc,
			Names: []*ast.Ident{ast.NewIdent(capitalCase(name))},
			Type:  f.structMemberGetterFuncType(name, member),
		}
		setter := &ast.Field{
			Names: []*ast.Ident{ast.NewIdent("Set" + capitalCase(name))},
			Type:  f.structMemberSetterFuncType(name, member),
		}
		list = append(list, setter)
		list = append(list, getter)
	}

	fields := &ast.FieldList{
		List: list,
	}
	return &ast.InterfaceType{
		Methods: fields,
	}
}

func (f *modelFile) structMemberIdent(name string) string {
	//We prepend the following string to force the field to be exported. The ! below is an upper-case unicode character
	return "Ị_" + name
}

func (f *modelFile) structMemberGetterFuncType(name string, member shape.Reference) *ast.FuncType {
	return &ast.FuncType{
		Results: fieldList(field("", f.publicType(member))),
	}
}

func (f *modelFile) structMemberSetterFuncType(name string, member shape.Reference) *ast.FuncType {
	return &ast.FuncType{
		Params: fieldList(field("v", f.publicType(member))),
	}
}

func (f *modelFile) structImplFields(structure *shape.Structure) []*ast.Field {
	list := make([]*ast.Field, 0)
	if isa := structure.IsA; isa != nil {
		if structure.IsException {
			//emit SuperType
			list = append(list, field("", f.publicType(*isa)))
		}

		super, err := f.allAsms.LookupShape(*isa)
		if err != nil {
			panic(err)
		}
		if super, ok := super.(*shape.Structure); ok {
			list = append(list, f.structImplFields(super)...)
		} else {
			panic("Cannot inherit from a non-structure shape")
		}
	}
	for name, member := range structure.Members {
		newField := field(f.structMemberIdent(name), f.publicType(member))
		//Provide the original name of the struct member so encoding/decoding is possible
		newField.Tag = &ast.BasicLit{
			Kind:  token.STRING,
			Value: "`coral:\"" + name + "\" json:\"" + name + "\"`",
		}
		list = append(list, newField)
	}
	return list
}

//Generates the type expression for the impl of a coral struct
//The struct implements the interface emitted by structTypeExpr
func (f *modelFile) structImplExpr(structure *shape.Structure) ast.Expr {
	list := f.structImplFields(structure)
	return &ast.StructType{
		Fields: fieldList(list...),
	}
}

//Decls for only the members of the structure.
func (f *modelFile) structImplMemberDecls(recv *ast.FieldList, structure *shape.Structure) []ast.Decl {
	decls := make([]ast.Decl, 0, len(structure.Members))
	if isa := structure.IsA; isa != nil {
		//Can't use struct embedding because the JSON package ignores them.
		//So, we have to provide our own Getters & Setters for the super type
		super, err := f.allAsms.LookupShape(*isa)
		if err != nil {
			//Invalid or unknown reference
			panic(err)
		}

		if super, ok := super.(*shape.Structure); !ok {
			panic("Cannot inherit from a non-structure type")
		} else {
			//Add in all of the members of the super type.
			decls = append(decls, f.structImplMemberDecls(recv, super)...)
		}
	}

	if structure.IsException && structure.IsA == nil {
		//TODO: Verify that we only need to emit the error method on the base class for exceptions.

		errorBody := blockStmt(returnStmt(
			__model__.ErrorMessage("this"),
		))

		errorFun := &ast.FuncDecl{
			Recv: recv,
			Name: ast.NewIdent("Error"),
			Type: &ast.FuncType{
				Results: fieldList(field("", "string")),
			},
			Body: errorBody,
		}
		decls = append(decls, errorFun)
	}

	for name, member := range structure.Members {
		thisMember := &ast.BinaryExpr{
			X:  ast.NewIdent("this"),
			Op: token.PERIOD,
			Y:  ast.NewIdent(f.structMemberIdent(name)),
		}

		//return this.foo
		getterBody := blockStmt(returnStmt(thisMember))
		getter := &ast.FuncDecl{
			Recv: recv,
			Name: ast.NewIdent(capitalCase(name)),
			Type: f.structMemberGetterFuncType(name, member),
			Body: getterBody,
		}

		//this.foo = foo
		setterBody := blockStmt(&ast.AssignStmt{
			Lhs: []ast.Expr{thisMember},
			Tok: token.ASSIGN,
			Rhs: []ast.Expr{f.structMemberSetterFuncType(name, member).Params.List[0].Names[0]},
		})
		setter := &ast.FuncDecl{
			Recv: recv,
			Name: ast.NewIdent("Set" + capitalCase(name)),
			Type: f.structMemberSetterFuncType(name, member),
			Body: setterBody,
		}
		decls = append(decls, getter)
		decls = append(decls, setter)
	}
	return decls
}

func (f *modelFile) structImplDecls(structure *shape.Structure) []ast.Decl {
	decls := make([]ast.Decl, 0, len(structure.Members))
	recv := fieldList(
		field("this", "*"+f.privateIdent(structure.Name())),
	)
	/*
	   struct {

	       _foo Foo
	   }
	*/
	decls = append(decls, f.structImplMemberDecls(recv, structure)...)

	newFuncIdent := f.factoryIdent(structure.Name())
	//New() method
	{
		t := &ast.FuncType{
			Results: fieldList(
				field("", f.publicType(structure.Name())),
			),
		}
		body := blockStmt(
			returnStmt(
				&ast.UnaryExpr{
					Op: token.AND,
					X: &ast.CompositeLit{
						Type: ast.NewIdent(f.privateIdent(structure.Name())),
					},
				},
			),
		)

		fun := &ast.FuncDecl{
			Name: ast.NewIdent(newFuncIdent),
			Type: t,
			Body: body,
		}
		decls = append(decls, fun)
	}

	//init() method
	{
		t := &ast.FuncType{}
		// Collect statements for registering shapes to all target services
		serviceImpls := make([]ast.Stmt, 0)
		// var val Foo
		serviceImpls = append(serviceImpls, declare_var("val", f.publicType(structure.Name())))
		// t := __reflect__.TypeOf(&val)
		serviceImpls = append(serviceImpls, define_var("t", __reflect__.TypeOf("val")))
		for _, service := range f.targetSvcs {
			// __model__.LookupService(svc).Assembly(asm).RegisterShape(shape, t, func() interface {} { return NewFoo() })
			serviceImpls = append(
				serviceImpls,
				&ast.ExprStmt{
					X: &ast.BinaryExpr{
						X: &ast.BinaryExpr{
							X:  __model__.LookupService(service.Name().Name()),
							Op: token.PERIOD,
							Y:  __model__.Assembly(f.asm.Namespace),
						},
						Op: token.PERIOD,
						Y: __model__.RegisterShape(structure.Name().Name(), "t", &ast.FuncLit{
							Type: &ast.FuncType{
								Results: fieldList(
									&ast.Field{
										Type: ast.NewIdent("interface{}"),
									},
								),
							},
							Body: blockStmt(
								returnStmt(
									&ast.CallExpr{
										Fun: ast.NewIdent(newFuncIdent),
									},
								),
							),
						}),
					},
				},
			)
		}
		body := blockStmt(serviceImpls...)
		fun := &ast.FuncDecl{
			Name: ast.NewIdent("init"),
			Type: t,
			Body: body,
		}
		decls = append(decls, fun)
	}
	return decls
}

func (f *modelFile) operationTypeExpr(operation *shape.Operation) ast.Expr {
	//Operations do not have type
	return nil
}

// serviceTypeExpr constructs an interface definition for the service operations.
func (f *modelFile) serviceTypeExpr(service *shape.Service) ast.Expr {
	fields := make([]*ast.Field, 0, len(service.Operations))
	for _, op := range service.Operations {
		fields = append(fields, f.operationDecl(op))
	}
	return &ast.InterfaceType{
		Methods:    &ast.FieldList{List: fields},
		Incomplete: false,
	}
}

// operationDecl generates a declaration of the form:
// 		// Comment
// 		Operation(input InputType) (OutputType, error)
// where the Comment, InputType and OutputType are optional.
func (f *modelFile) operationDecl(operation *shape.Operation) *ast.Field {
	input := &ast.FieldList{}
	output := fieldList(field("", "error"))
	if operation.Input != nil {
		input.List = []*ast.Field{
			field("input", f.asmFile().publicType(operation.Input)),
		}
	}

	if operation.Output != nil {
		newField := field("", f.asmFile().publicType(operation.Output))
		output.List = append([]*ast.Field{newField}, output.List...)
	}

	return &ast.Field{
		Doc:   comment(operation.Doc()),
		Names: []*ast.Ident{ast.NewIdent(f.publicIdent(operation.Name()))},
		Type: &ast.FuncType{
			Params:  input,
			Results: output,
		},
	}
}
