/* Copyright 2016 Amazon.com, Inc. or its affiliates. All Rights Reserved. */
package golang

import (
	"coral/shape"
	"go/ast"
	"go/token"
)

// typeExpr satisfies fileGen.
// Generates the service client struct.
func (f *serverFile) typeExpr(s shape.Interface) ast.Expr {
	switch i := s.(type) {
	case *shape.Service:
		return f.serviceTypeExpr(i)
	}
	return nil
}

// implExpr satisfies fileGen.
// Returns nil
func (f *serverFile) implExpr(s shape.Interface) ast.Expr {
	// no impls
	return nil
}

// serviceTypeExpr generates the service server struct with an empty
// field list.
func (f *serverFile) serviceTypeExpr(s *shape.Service) ast.Expr {
	return &ast.StructType{
		Fields: &ast.FieldList{
			List: []*ast.Field{},
		},
	}
}

// importSpecs satisfies fileGen.
// Generates imports for coral/client, coral/codec, and coral/dialer.
func (f *serverFile) importSpecs(s shape.Interface) []*ast.ImportSpec {
	switch s.(type) {
	case *shape.Service:
		specs := []*ast.ImportSpec{
			{
				Name: ast.NewIdent("_ctx_"),
				Path: &ast.BasicLit{
					Kind:  token.STRING,
					Value: `"context"`,
				},
			},
		}
		return specs
	}
	return nil
}

// implDecls satisfies fileGen.
// Generates function implementations.
func (f *serverFile) implDecls(s shape.Interface) []ast.Decl {
	switch i := s.(type) {
	case *shape.Service:
		return f.serviceImplDecls(i)
	}
	return nil
}

// serviceImplDecls generates a client builder and function implementations for
// each of the service operations.
func (f *serverFile) serviceImplDecls(s *shape.Service) []ast.Decl {
	decls := make([]ast.Decl, 0, len(s.Operations))
	for _, op := range s.Operations {
		t := f.ptrIdent(s.Name())
		decls = append(decls, f.operationImplDecl(t, op))
	}
	return decls
}

// ptrIdent returns a string that represents a pointer to the public
// identity of the client.
func (f *serverFile) ptrIdent(r shape.Reference) string {
	return "*" + f.publicIdent(r)
}

// publicIdent satisfies fileGen.
// Returns a string that represents the public identity of the client.
func (f *serverFile) publicIdent(r shape.Reference) string {
	return f.asmFile().publicIdent(r) + "Server"
}

// operationImplDecl generates the implementation of an operation with logic in the form:
// 		return nil, nil
// With variations based on if output is missing.
func (f *serverFile) operationImplDecl(t string, o *shape.Operation) ast.Decl {
	thisIdent := "this"
	ctxIdent := "ctx"
	inputIdent := "input"
	outputIdent := "output"
	nilIdent := "nil"
	input := &ast.FieldList{}
	output := fieldList(field("", "error"))
	input.List = []*ast.Field{field(ctxIdent, "_ctx_.Context")}
	if o.Input != nil {
		input.List = append(input.List, field(inputIdent, f.asmFile().publicType(o.Input)))
	} else {
		inputIdent = ""
	}

	if o.Output != nil {
		newField := field("", f.asmFile().publicType(o.Output))
		output.List = append([]*ast.Field{newField}, output.List...)
	} else {
		outputIdent = ""
	}

	body := blockStmt()

	// Return statements.  Either return a placeholder for the output and error,
	// or only a placeholder for error.
	// TODO: We should probably return an error that says "Not implemented".
	if outputIdent != "" {
		// return nil, nil
		stmt := returnStmt(
			ast.NewIdent(nilIdent),
			ast.NewIdent(nilIdent),
		)
		body.List = append(body.List, stmt)
	} else {
		// return nil
		stmt := returnStmt(ast.NewIdent(nilIdent))
		body.List = append(body.List, stmt)
	}

	return &ast.FuncDecl{
		Doc:  comment(o.Doc()),
		Recv: fieldList(field(thisIdent, t)),
		Name: ast.NewIdent(f.asmFile().publicIdent(o.Name())),
		Type: &ast.FuncType{
			Params:  input,
			Results: output,
		},
		Body: body,
	}
}
