/* Copyright 2016 Amazon.com, Inc. or its affiliates. All Rights Reserved. */
package golang

import (
	"coral/shape"
	"go/ast"
	"go/token"
	"strings"

	"github.com/pkg/errors"
)

// implExpr returns the type expression for a private type that implements the shape.
func (f *serverModelFile) implExpr(iface shape.Interface) ast.Expr {
	switch s := iface.(type) {
	case *shape.Structure:
		return f.structImplExpr(s)
	}
	return nil
}

// typeExpr returns the type expression that defines the shape.
func (f *serverModelFile) typeExpr(iface shape.Interface) ast.Expr {
	if nativeType := f.nativeType(iface); nativeType != nil {
		//The type can be represented natively in Go
		return nil
	}
	switch s := iface.(type) {
	case *shape.Structure:
		return f.structTypeExpr(s)
	case *shape.Operation:
		return f.operationTypeExpr(s)
	case *shape.Service:
		return f.serviceTypeExpr(s)
	}
	panic(errors.Errorf("Bug: Unknown type %T", iface))
}

func (f *serverModelFile) implDecls(iface shape.Interface) []ast.Decl {
	switch s := iface.(type) {
	case *shape.Structure:
		return f.structImplDecls(s)
	case *shape.Operation:
		return f.operationImplDecls(s)
	case *shape.Service:
		return nil
	}
	return f.genericImplDecls(iface)
}

func (f *serverModelFile) genericImplDecls(s shape.Interface) []ast.Decl {
	return nil
}

// importSpecs adds the import statements based upon what types of
// shapes are found.  It is assumed that import deduplication is
// handled by the caller.
func (f *serverModelFile) importSpecs(iface shape.Interface) []*ast.ImportSpec {
	seen := map[shape.Interface]bool{}
	var specFunc func(iface shape.Interface) []string
	specFunc = func(iface shape.Interface) []string {
		paths := []string{}
		if _, ok := seen[iface]; ok {
			// Prevent infinite loop for circular dependencies.
			return paths
		}
		seen[iface] = true
		switch iface := iface.(type) {
		case *shape.BigInteger, *shape.BigDecimal:
			paths = append(paths, "math/big")
		case *shape.Timestamp:
			paths = append(paths, "time")
		case *shape.Structure:
			paths = append(paths, "reflect", "coral/model")
			for _, ref := range iface.Members {
				shp, err := f.allAsms.LookupShape(ref)
				if err != nil {
					continue
				}
				paths = append(paths, specFunc(shp)...)
			}
		case *shape.Service:
			// Some services have only references and are empty.
			if len(iface.Operations) != 0 {
				paths = append(paths, "reflect", "coral/model", "context")
			}
		default:
			paths = append(paths, "reflect", "coral/model")
		}
		return paths
	}
	paths := specFunc(iface)

	specs := make([]*ast.ImportSpec, 0, len(paths))
	for _, path := range paths {
		paths = strings.Split(path, "/")
		path = `"` + path + `"`
		specs = append(specs, &ast.ImportSpec{
			// Change every import foo to __foo__
			Name: ast.NewIdent("__" + paths[len(paths)-1] + "__"),
			Path: &ast.BasicLit{
				Kind:  token.STRING,
				Value: path,
			},
		})
	}
	return specs
}

func (f *serverModelFile) structTypeExpr(structure *shape.Structure) ast.Expr {
	/*
		type Thing interface {
			[SuperType Methods]
			Foo() FooType
			SetFoo(FooType f)
		}
	*/
	// 2*len for Getters & Setters
	list := make([]*ast.Field, 0, 2*len(structure.Members))

	if isa := structure.IsA; isa != nil {
		// Emit SuperType methods.
		// If the SuperType is located in a different package, embedding will not work (cannot implement __type())
		super, err := f.allAsms.LookupShape(*isa)
		if err != nil {
			panic(err)
		}
		if super, ok := super.(*shape.Structure); ok {
			list = append(list, f.structTypeExpr(super).(*ast.InterfaceType).Methods.List...)
		}
	}

	if structure.IsException && structure.IsA == nil {
		errorField := &ast.Field{
			Type: ast.NewIdent("error"),
		}
		list = append(list, errorField)
	}
	// Getters and setters.
	for name, member := range structure.Members {
		doc := comment(member.Doc)
		getter := &ast.Field{
			Doc:   doc,
			Names: []*ast.Ident{ast.NewIdent(capitalCase(name))},
			Type:  f.structMemberGetterFuncType(name, member),
		}
		setter := &ast.Field{
			Names: []*ast.Ident{ast.NewIdent("Set" + capitalCase(name))},
			Type:  f.structMemberSetterFuncType(name, member),
		}
		list = append(list, getter)
		list = append(list, setter)
	}

	fields := &ast.FieldList{
		List: list,
	}
	return &ast.InterfaceType{
		Methods: fields,
	}
}

func (f *serverModelFile) structMemberIdent(name string) string {
	// We prepend the following string to force the field to be exported.
	// The Ị below is an upper-case unicode character.
	return "Ị_" + name
}

func (f *serverModelFile) structMemberGetterFuncType(name string, member shape.Reference) *ast.FuncType {
	return &ast.FuncType{
		Results: fieldList(field("", f.publicType(member))),
	}
}

func (f *serverModelFile) structMemberSetterFuncType(name string, member shape.Reference) *ast.FuncType {
	return &ast.FuncType{
		Params: fieldList(field("v", f.publicType(member))),
	}
}

func (f *serverModelFile) structImplFields(structure *shape.Structure) []*ast.Field {
	list := make([]*ast.Field, 0)
	if isa := structure.IsA; isa != nil {
		if structure.IsException {
			// Emit SuperType.
			list = append(list, field("", f.publicType(*isa)))
		}

		super, err := f.allAsms.LookupShape(*isa)
		if err != nil {
			panic(err)
		}
		if super, ok := super.(*shape.Structure); ok {
			list = append(list, f.structImplFields(super)...)
		} else {
			panic("Cannot inherit from a non-structure shape")
		}
	}
	for name, member := range structure.Members {
		newField := field(f.structMemberIdent(name), f.publicType(member))
		// Provide the original name of the struct member so encoding/decoding is possible.
		newField.Tag = &ast.BasicLit{
			Kind:  token.STRING,
			Value: "`coral:\"" + name + "\" json:\"" + name + "\"`",
		}
		list = append(list, newField)
	}
	return list
}

// structImplExpr generates the type expression for the impl of a coral struct.
// The struct implements the interface emitted by structTypeExpr.
func (f *serverModelFile) structImplExpr(structure *shape.Structure) ast.Expr {
	/*
		type _Thing struct {
			Ị_foo FooType `coral:"foo" json:"foo"`
		}
	*/
	list := f.structImplFields(structure)
	return &ast.StructType{
		Fields: fieldList(list...),
	}
}

// structImplMemberDecls generates Decls for only the members of the structure.
func (f *serverModelFile) structImplMemberDecls(recv *ast.FieldList, structure *shape.Structure) []ast.Decl {
	// 2*len for Getters & Setters
	decls := make([]ast.Decl, 0, 2*len(structure.Members))
	if isa := structure.IsA; isa != nil {
		// Can't use struct embedding because the JSON package ignores them.
		// We have to provide our own Getters & Setters for the super type.
		super, err := f.allAsms.LookupShape(*isa)
		if err != nil {
			// Invalid or unknown reference.
			panic(err)
		}

		if super, ok := super.(*shape.Structure); !ok {
			panic("Cannot inherit from a non-structure type")
		} else {
			// Add in all of the members of the super type.
			decls = append(decls, f.structImplMemberDecls(recv, super)...)
		}
	}

	if structure.IsException && structure.IsA == nil {
		errorBody := blockStmt(returnStmt(
			__model__.ErrorMessage("this"),
		))

		errorFun := &ast.FuncDecl{
			Recv: recv,
			Name: ast.NewIdent("Error"),
			Type: &ast.FuncType{
				Results: fieldList(field("", "string")),
			},
			Body: errorBody,
		}
		decls = append(decls, errorFun)
	}

	/*
		Getters and setters:
			func (this *_Thing) Foo() FooType {
				return this.Ị_foo
			}

			func (this *_Thing) SetFoo(v FooType) {
				this.Ị_foo = v
			}
	*/
	for name, member := range structure.Members {
		thisMember := &ast.BinaryExpr{
			X:  ast.NewIdent("this"),
			Op: token.PERIOD,
			Y:  ast.NewIdent(f.structMemberIdent(name)),
		}

		// return this.foo
		getterBody := blockStmt(returnStmt(thisMember))
		getter := &ast.FuncDecl{
			Recv: recv,
			Name: ast.NewIdent(capitalCase(name)),
			Type: f.structMemberGetterFuncType(name, member),
			Body: getterBody,
		}

		// this.foo = foo
		setterBody := blockStmt(&ast.AssignStmt{
			Lhs: []ast.Expr{thisMember},
			Tok: token.ASSIGN,
			Rhs: []ast.Expr{f.structMemberSetterFuncType(name, member).Params.List[0].Names[0]},
		})
		setter := &ast.FuncDecl{
			Recv: recv,
			Name: ast.NewIdent("Set" + capitalCase(name)),
			Type: f.structMemberSetterFuncType(name, member),
			Body: setterBody,
		}
		decls = append(decls, getter)
		decls = append(decls, setter)
	}
	return decls
}

func (f *serverModelFile) structImplDecls(structure *shape.Structure) []ast.Decl {
	// 2 for New and init, 2*len for Getters & Setters
	decls := make([]ast.Decl, 0, 2+2*len(structure.Members))
	recv := fieldList(
		field("this", "*"+f.privateIdent(structure.Name())),
	)

	// Getters and setters.
	decls = append(decls, f.structImplMemberDecls(recv, structure)...)

	/*
		func NewFoo() Foo {
			return &_Foo{}
		}
	*/
	newFuncIdent := f.factoryIdent(structure.Name())
	{
		t := &ast.FuncType{
			Results: fieldList(
				field("", f.publicType(structure.Name())),
			),
		}
		body := blockStmt(
			returnStmt(
				&ast.UnaryExpr{
					Op: token.AND,
					X: &ast.CompositeLit{
						Type: ast.NewIdent(f.privateIdent(structure.Name())),
					},
				},
			),
		)

		fun := &ast.FuncDecl{
			Name: ast.NewIdent(newFuncIdent),
			Type: t,
			Body: body,
		}
		decls = append(decls, fun)
	}

	/*
		func init() {
			var val Foo
			t := __reflect__.TypeOf(&val)
			__model__.LookupAssembly(asm).RegisterShape(shape, t, func() interface{} {
				return NewFoo()
			})
		}
	*/
	{
		t := &ast.FuncType{}
		serviceImpls := make([]ast.Stmt, 0)
		// var val Foo
		serviceImpls = append(serviceImpls, declare_var("val", f.publicType(structure.Name())))
		// t := __reflect__.TypeOf(&val)
		serviceImpls = append(serviceImpls, define_var("t", __reflect__.TypeOf("val")))
		for _, service := range f.targetSvcs {
			// __model__.LookupService(svc).Assembly(asm).RegisterShape(shape, t, func() interface {} { return NewFoo() })
			serviceImpls = append(
				serviceImpls,
				&ast.ExprStmt{
					X: &ast.BinaryExpr{
						X: &ast.BinaryExpr{
							X:  __model__.LookupService(service.Name().Name()),
							Op: token.PERIOD,
							Y:  __model__.Assembly(f.asm.Namespace),
						},
						Op: token.PERIOD,
						Y: __model__.RegisterShape(structure.Name().Name(), "t", &ast.FuncLit{
							Type: &ast.FuncType{
								Results: fieldList(
									&ast.Field{
										Type: ast.NewIdent("interface{}"),
									},
								),
							},
							Body: blockStmt(
								returnStmt(
									&ast.CallExpr{
										Fun: ast.NewIdent(newFuncIdent),
									},
								),
							),
						}),
					},
				},
			)
		}
		body := blockStmt(serviceImpls...)
		fun := &ast.FuncDecl{
			Name: ast.NewIdent("init"),
			Type: t,
			Body: body,
		}
		decls = append(decls, fun)
	}
	return decls
}

// operationImplDecls registers the given operation in an init() function.
func (f *serverModelFile) operationImplDecls(operation *shape.Operation) []ast.Decl {
	/*
		func init() {
			svc := __model__.LookupService(svc)
			input, _ := svc.Assembly(inputNamespace).Shape(inputName)
			output, _ := svc.Assembly(outputNamespace)..Shape(outputName)
			svc.Assembly(namespace).RegisterOp(opName, input, output, nil)
		}
	*/
	decls := make([]ast.Decl, 0, 1)
	serviceOperations := make([]ast.Stmt, 0)
	for _, service := range operation.Services {
		svcName := service.Name()
		svcIdent := ast.NewIdent("svc" + svcName)
		serviceOperations = append(
			serviceOperations,
			// svc := __model__.LookupService(svc)
			define_var(svcIdent.Name, __model__.LookupService(svcName)),
		)
		serviceOperations = append(
			serviceOperations,
			// input, _ := svc.Assembly(inputNamespace).Shape(inputName)
			f.genStmt("input", svcIdent, operation.Input),
			// output, _ := svc.Assembly(outputNamespace)..Shape(outputName)
			f.genStmt("output", svcIdent, operation.Output),
			// svc.Assembly(namespace).RegisterOp(opName, input, output, nil)
			&ast.ExprStmt{
				X: &ast.BinaryExpr{
					X:  svcAssembly(svcIdent, f.asm.Namespace),
					Op: token.PERIOD,
					Y:  __model__.RegisterOp(operation.Name().Name(), operation.Input, operation.Output, operation.Errors),
				},
			},
		)
	}
	body := blockStmt(serviceOperations...)
	fun := &ast.FuncDecl{
		Name: ast.NewIdent("init"),
		Type: &ast.FuncType{},
		Body: body,
	}
	decls = append(decls, fun)

	return decls
}

// svcAssembly generates a call of the form:
//
//	svcIdent.Assembly(namespace)
func svcAssembly(svcIdent *ast.Ident, namespace string) *ast.CallExpr {
	return &ast.CallExpr{
		Fun: ast.NewIdent(svcIdent.Name + ".Assembly"),
		Args: []ast.Expr{
			&ast.BasicLit{
				Kind:  token.STRING,
				Value: `"` + namespace + `"`,
			},
		},
	}
}

// genStmt return an empty statement if svIdent or r is nil, otherwise generates code of the form:
//
//	name, _ := svc.Assembly(r.Assembly).Shape(r.Name)
func (f *serverModelFile) genStmt(name string, svcIdent *ast.Ident, r shape.Reference) ast.Stmt {
	if svcIdent == nil || r == nil {
		return &ast.EmptyStmt{}
	}
	rhs := &ast.BinaryExpr{
		X:  svcAssembly(svcIdent, r.Assembly()),
		Op: token.PERIOD,
		Y: &ast.CallExpr{
			Fun: ast.NewIdent("Shape"),
			Args: []ast.Expr{
				&ast.BasicLit{
					Kind:  token.STRING,
					Value: `"` + r.Name() + `"`,
				},
			},
		},
	}

	return define_vars([]string{name, "_"}, rhs)
}

func (f *serverModelFile) operationTypeExpr(operation *shape.Operation) ast.Expr {
	// Operations do not have type.
	return nil
}

// serviceTypeExpr constructs an interface definition for the service operations.
func (f *serverModelFile) serviceTypeExpr(service *shape.Service) ast.Expr {
	fields := make([]*ast.Field, 0, len(service.Operations))
	for _, op := range service.Operations {
		fields = append(fields, f.operationDecl(op))
	}
	return &ast.InterfaceType{
		Methods:    &ast.FieldList{List: fields},
		Incomplete: false,
	}
}

// operationDecl generates a declaration of the form:
// 		// Comment
// 		Operation(input InputType) (OutputType, error)
// where the Comment, InputType and OutputType are optional.
func (f *serverModelFile) operationDecl(operation *shape.Operation) *ast.Field {
	input := &ast.FieldList{}
	output := fieldList(field("", "error"))
	input.List = []*ast.Field{field("ctx", "__context__.Context")}
	if operation.Input != nil {
		input.List = append(input.List, field("input", f.asmFile().publicType(operation.Input)))
	}

	if operation.Output != nil {
		newField := field("", f.asmFile().publicType(operation.Output))
		output.List = append([]*ast.Field{newField}, output.List...)
	}

	return &ast.Field{
		Doc:   comment(operation.Doc()),
		Names: []*ast.Ident{ast.NewIdent(f.publicIdent(operation.Name()))},
		Type: &ast.FuncType{
			Params:  input,
			Results: output,
		},
	}
}
