package golang

import (
	"coral/shape"
	"github.com/pkg/errors"
	"go/ast"
)

func (f *assemblyFile) publicType(r shape.Reference) ast.Expr {
	native := f.nativeType(r)
	if native != nil {
		return native
	}
	return ast.NewIdent(f.publicIdent(r))
}

func (f *assemblyFile) publicValueType(r shape.Reference) ast.Expr {
	native := f.nativeValueType(r)
	if native != nil {
		return native
	}
	return ast.NewIdent(f.publicIdent(r))
}

//Get the interface into a shape.Interface
func (f *assemblyFile) resolve(i interface{}) shape.Interface {
	if r, ok := i.(shape.Reference); ok {
		iface, err := f.allAsms.LookupShape(r)
		if err != nil {
			panic(err)
		}
		return iface
	} else if i, ok := i.(shape.Interface); ok {
		return i
	} else {
		panic(errors.Errorf("Unknown Type %T", i))
	}
}

func (f *assemblyFile) nativeType(i interface{}) ast.Expr {
	iface := f.resolve(i)

	switch s := iface.(type) {
	case *shape.Boolean:
		return ast.NewIdent("*bool")
	case *shape.Blob:
		return &ast.ArrayType{
			Elt: ast.NewIdent("byte"),
		}
	case *shape.Timestamp:
		return ast.NewIdent("*__time__.Time")
	case *shape.Byte:
		return ast.NewIdent("*int8")
	case *shape.Character:
		return ast.NewIdent("*uint8")
	case *shape.Short:
		return ast.NewIdent("*int16")
	case *shape.Integer:
		return ast.NewIdent("*int32")
	case *shape.Long:
		return ast.NewIdent("*int64")
	case *shape.String:
		return ast.NewIdent("*string")
	case *shape.Envelope:
		return ast.NewIdent("*struct{}")
	case *shape.BigInteger:
		return ast.NewIdent("*__big__.Int")
	case *shape.BigDecimal:
		return ast.NewIdent("*__big__.Rat")
	case *shape.Float:
		return ast.NewIdent("*float32")
	case *shape.Double:
		return ast.NewIdent("*float64")
	case *shape.Map:
		return f.mapTypeExpr(s)
	case *shape.List:
		return f.listTypeExpr(s)
	}
	return nil
}

func (f *assemblyFile) nativeValueType(i interface{}) ast.Expr {
	iface := f.resolve(i)

	switch s := iface.(type) {
	case *shape.Boolean:
		return ast.NewIdent("bool")
	case *shape.Blob:
		return &ast.ArrayType{
			Elt: ast.NewIdent("byte"),
		}
	case *shape.Timestamp:
		return ast.NewIdent("__time__.Time")
	case *shape.Byte:
		return ast.NewIdent("int8")
	case *shape.Short:
		return ast.NewIdent("int16")
	case *shape.Character:
		return ast.NewIdent("unit8")
	case *shape.Integer:
		return ast.NewIdent("int32")
	case *shape.Long:
		return ast.NewIdent("int64")
	case *shape.String:
		return ast.NewIdent("string")
	case *shape.Envelope:
		return ast.NewIdent("struct{}")
	case *shape.BigInteger:
		return ast.NewIdent("__big__.Int")
	case *shape.BigDecimal:
		return ast.NewIdent("__big__.Rat")
	case *shape.Float:
		return ast.NewIdent("float32")
	case *shape.Double:
		return ast.NewIdent("float64")
	case *shape.Map:
		return f.mapTypeExpr(s)
	case *shape.List:
		return f.listTypeExpr(s)
	}
	return nil
}

func (f *assemblyFile) listTypeExpr(list *shape.List) ast.Expr {
	return &ast.ArrayType{
		// Use publicValueType so that the slice values aren't pointers.
		// Absence of value is represented by the slice itself being nil.
		Elt: f.publicValueType(list.Member),
	}
}

func (f *assemblyFile) mapTypeExpr(m *shape.Map) ast.Expr {
	return &ast.MapType{
		// Use publicValueType so that the keys aren't pointers.
		Key:   f.publicValueType(m.Key),
		Value: f.publicType(m.Value),
	}
}
